<?php

namespace Drupal\affiliated;

use Drupal\affiliated\Entity\AffiliateCampaignInterface;
use Drupal\affiliated\Event\AffiliateAccountLookupEvent;
use Drupal\affiliated\Event\AffiliateCodeLookupEvent;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class AffiliateManager service.
 */
class AffiliateManager {

  use StringTranslationTrait;

  /**
   * The EntityTypeManagerInterface.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user account interface.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The request stack.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * The affiliate config settings.
   *
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Contracts\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * Constructs an AffiliateManager service.
   */
  public function __construct(
    AccountInterface $currentUser,
    EntityTypeManagerInterface $entity_type_manager,
    RequestStack $request_stack,
    ModuleHandler $module_handler,
    ConfigFactory $config_factory,
    EventDispatcherInterface $event_dispatcher,
  ) {
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
    $this->eventDispatcher = $event_dispatcher;
    $this->moduleHandler = $module_handler;
    $this->config = $config_factory->get('affiliated.settings');
  }

  /**
   * Checks if a user account is an active affiliate.
   *
   * This can be called multiple times per request so statically caching.
   */
  public function isActiveAffiliate(AccountInterface $account) {
    static $access;
    if (isset($access[$account->id()])) {
      return $access[$account->id()];
    }
    $is_affiliate = $account->hasPermission('act as an affiliate');
    // Allow other modules to decide if this is a valid affiliate.
    $this->moduleHandler->alter('affiliated_active_affiliate', $is_affiliate, $account);
    $access[$account->id()] = $is_affiliate;
    return $access[$account->id()];
  }

  /**
   * Returns the default global campaign.
   *
   * The default campaign is created during installation. All clicks with no
   * campaign id specified (like ref/%user) are attributed to this campaign.
   *
   * @return \Drupal\affiliated\Entity\AffiliateCampaignInterface|null
   *   The campaign entity.
   */
  public function getDefaultCampaign() {
    $campaignStorage = $this->entityTypeManager->getStorage('affiliate_campaign');
    $id = $campaignStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('is_global', 1)
      ->condition('is_default', 1)
      ->range(0, 1)
      ->execute();

    if ($id) {
      return $campaignStorage->load(reset($id));
    }
    return NULL;
  }

  /**
   * Registers a click.
   *
   * Creates click entity if store_clicks is enabled.
   *
   * @param \Drupal\Core\Session\AccountInterface $affiliate
   *   The user object of the affiliate getting the click.
   * @param \Drupal\affiliated\Entity\AffiliateCampaignInterface $campaign
   *   The campaign entity.
   * @param string $destination
   *   The end destination (requested URL).
   * @param string|null $referrer
   *   The referring URL, or NULL to use the HTTP referer header.
   *
   * @return \Drupal\affiliated\Entity\AffiliateClickInterface|bool
   *   The click entity if stored, TRUE if click storage is disabled but
   *   tracking should proceed, or FALSE if the click was rejected.
   */
  public function registerClick(AccountInterface $affiliate, AffiliateCampaignInterface $campaign, string $destination, ?string $referrer = NULL) {
    // Check if the affiliate is the same as the current user.
    if (!$this->config->get('allow_owner') && $affiliate->id() == $this->currentUser->id()) {
      return FALSE;
    }

    // If click storage is disabled, return TRUE to allow cookie to be set.
    if (!$this->config->get('store_clicks')) {
      return TRUE;
    }

    $request = $this->requestStack->getCurrentRequest();
    // Register the click.
    $click = $this->entityTypeManager->getStorage('affiliate_click')->create([
      'campaign' => $campaign->id(),
      'affiliate' => $affiliate->id(),
      'hostname' => $request->getClientIp(),
      'referrer' => $referrer ?? $request->server->get('HTTP_REFERER'),
      'destination' => $destination,
    ]);
    $click->save();
    return $click;
  }

  /**
   * Gets the affiliate code from the cookie.
   *
   * @return string|null
   *   The affiliate code, or NULL if not set.
   */
  public function getStoredAffiliateCode() {
    return $this->requestStack->getCurrentRequest()->cookies->get('affiliate_id');
  }

  /**
   * Gets the affiliates user account from the cookie.
   *
   * @return \Drupal\user\UserInterface|null
   *   The affiliates user entity or null
   */
  public function getStoredAccount() {
    if ($affiliate_code = $this->getStoredAffiliateCode()) {
      return $this->getAccountFromCode($affiliate_code);
    }
    return NULL;
  }

  /**
   * Gets the user account from the code variable.
   *
   * The code could either be the user id or the username depending on the
   * config settings.
   *
   * @param string|int $account_code
   *   Either a username or a user_id.
   *
   * @return \Drupal\user\UserInterface|null
   *   The user account if it is an active affiliate.
   */
  public function getAccountFromCode($account_code) {
    $user = NULL;
    if ($account_code) {
      switch ($this->config->get('affiliate_code_type')) {
        case 'username':
          $user = user_load_by_name($account_code);
          break;

        case 'user_id':
        default:
          if (is_numeric($account_code)) {
            $user = $this->entityTypeManager->getStorage('user')->load($account_code);
          }
          break;
      }

      // Dispatch the account lookup event.
      // Allows other modules to alter the user account returned from the code.
      $event = new AffiliateAccountLookupEvent($account_code, $user);
      $this->eventDispatcher->dispatch($event, AffiliateAccountLookupEvent::EVENT_NAME);
      $user = $event->getAccount();

      // Check that this account is a valid affiliate.
      if ($user && $this->isActiveAffiliate($user)) {
        return $user;
      }
    }

    return NULL;
  }

  /**
   * Gets the affiliate code from a drupal account.
   */
  public function getCodeFromAccount($account) {
    switch ($this->config->get('affiliate_code_type')) {
      case 'username':
        $affiliate_code = $account->getAccountName();
        break;

      case 'user_id':
      default:
        $affiliate_code = $account->id();
        break;
    }

    // Dispatch the account lookup event.
    // Allows other modules to alter the affiliate code returned from the
    // account.
    $event = new AffiliateCodeLookupEvent($account, $affiliate_code);
    $this->eventDispatcher->dispatch($event, AffiliateCodeLookupEvent::EVENT_NAME);
    return $event->getCode();
  }

  /**
   * Gets the campaign code form the cookie.
   */
  public function getStoredCampaignCode() {
    return $this->requestStack->getCurrentRequest()->cookies->get('affiliate_campaign');
  }

  /**
   * Gets the campaign entity from the cookie.
   */
  public function getStoredCampaign() {
    $campaign_code = $this->getStoredCampaignCode();
    return $this->getCampaignFromCode($campaign_code);
  }

  /**
   * Gets the campaign entity from the code variable.
   *
   * Looks up by campaign code first (string), then falls back to entity ID
   * (numeric). Returns the default campaign if not found.
   *
   * @param string|int $campaign_code
   *   The campaign code or entity id.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The campaign entity or the default entity if not valid.
   */
  public function getCampaignFromCode($campaign_code) {
    if (!$campaign_code) {
      return $this->getDefaultCampaign();
    }

    $storage = $this->entityTypeManager->getStorage('affiliate_campaign');

    // If it's not numeric, look up by code field.
    if (!is_numeric($campaign_code)) {
      $ids = $storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('code', $campaign_code)
        ->condition('status', 1)
        ->range(0, 1)
        ->execute();
      if ($ids) {
        return $storage->load(reset($ids));
      }
    }
    else {
      // Numeric value - load by ID.
      /** @var \Drupal\affiliated\Entity\AffiliateCampaign $campaign */
      $campaign = $storage->load($campaign_code);
      if ($campaign && $campaign->isPublished()) {
        return $campaign;
      }
    }

    return $this->getDefaultCampaign();
  }

  /**
   * Sets up a conversion entity.
   *
   * Autopopulates the affiliate and campaign if the current user is cookied.
   * Otherwise, returns null.
   *
   * @param string $type
   *   The conversion bundle.
   *
   * @return \Drupal\affiliated\Entity\AffiliateConversion|null
   *   A new (unsaved) conversion with the referring affiliate and campaign
   *   populated or null if there is no referring affiliate account
   */
  public function createConversion($type) {
    $affiliate = $this->getStoredAccount();
    if (!$affiliate) {
      return NULL;
    }

    $campaign = $this->getStoredCampaign();

    // Validate campaign ownership - non-global campaigns can only be used
    // by their owner. This guards against cookie tampering.
    $campaign = $this->validateCampaignForAffiliate($campaign, $affiliate);

    $data = [
      'type' => $type,
      'affiliate' => $affiliate,
      'campaign' => $campaign,
    ];
    /** @var \Drupal\affiliated\Entity\AffiliateConversion $conversion */
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create($data);
    return $conversion;
  }

  /**
   * Validates that a campaign can be used by a specific affiliate.
   *
   * Non-global campaigns can only be used by their owner. If the campaign
   * is not valid for the affiliate, returns the default campaign instead.
   *
   * @param \Drupal\affiliated\Entity\AffiliateCampaignInterface|null $campaign
   *   The campaign to validate.
   * @param \Drupal\Core\Session\AccountInterface $affiliate
   *   The affiliate attempting to use the campaign.
   *
   * @return \Drupal\affiliated\Entity\AffiliateCampaignInterface|null
   *   The validated campaign, or the default campaign if invalid.
   */
  public function validateCampaignForAffiliate(?AffiliateCampaignInterface $campaign, AccountInterface $affiliate): ?AffiliateCampaignInterface {
    if (!$campaign) {
      return $this->getDefaultCampaign();
    }

    // Global campaigns can be used by anyone.
    if ($campaign->isGlobal()) {
      return $campaign;
    }

    // Non-global campaigns can only be used by their owner.
    if ($campaign->getOwnerId() == $affiliate->id()) {
      return $campaign;
    }

    // Not authorized to use this campaign - fall back to default.
    return $this->getDefaultCampaign();
  }

}
