<?php

namespace Drupal\affiliated\Entity\Handler;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * List builder for affiliate conversion entities.
 */
class AffiliateConversionListBuilder extends EntityListBuilder {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new AffiliateConversionListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, AccountProxyInterface $current_user) {
    parent::__construct($entity_type, $storage);
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    if ($this->currentUser->hasPermission('approve affiliate_conversion entities')) {
      if (!$entity->isPublished()) {
        $operations['approve'] = [
          'title' => $this->t('Approve'),
          'url' => $entity->toUrl('approve-form'),
          'weight' => 10,
        ];
      }
      else {
        $operations['cancel'] = [
          'title' => $this->t('Cancel'),
          'url' => $entity->toUrl('cancel-form'),
          'weight' => 10,
        ];
      }
    }

    return $operations;
  }

}
