/**
 * @file
 * Handles the A-Frame Inspector widget functionality.
 */

(function ($, Drupal, drupalSettings, once) {
  'use strict';

  Drupal.behaviors.aframeInspectorWidget = {
    attach: function (context, settings) {
      once('aframe-inspector', '.aframe-scene-textarea', context).forEach(function (textarea) {
        const $textarea = $(textarea);
        // Find the preview container - look up in the DOM tree
        const $fieldItem = $textarea.closest('.field--widget-aframe-inspector');
        const $preview = $fieldItem.find('.aframe-scene-preview').first();
        
        if (!$preview.length) {
          console.error('Could not find preview container for textarea:', $textarea);
          console.log('Field item:', $fieldItem);
          return;
        }

        let iframe = null;
        let updateTimeout = null;
        let isUpdatingFromIframe = false;

        // Function to create iframe HTML document
        function createIframeHTML(sceneHtml) {
          return `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>A-Frame Inspector</title>
  <script src="https://cdn.jsdelivr.net/npm/aframe@1.7.1/dist/aframe-master.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/aframe-inspector@1.7.1/dist/aframe-inspector.min.js"></script>
  <style>
    html, body { margin: 0; padding: 0; width: 100%; height: 100%; overflow: hidden; }
    a-scene { width: 100%; height: 100%; }
  </style>
</head>
<body>
  ${sceneHtml}
  <script>
    // Sync changes back to parent
    window.addEventListener('load', function() {
      const scene = document.querySelector('a-scene');
      if (scene) {
        let lastHTML = '';
        let pausePolling = false;
        
        // Listen for pause command from parent
        window.addEventListener('message', function(event) {
          if (event.data && event.data.type === 'pause-polling') {
            pausePolling = true;
            setTimeout(function() {
              pausePolling = false;
              lastHTML = scene.outerHTML; // Update baseline after pause
            }, 2000);
          }
        });
        
        // Make sure camera exists
        scene.addEventListener('loaded', function() {
          console.log('Scene loaded in iframe');
          if (!scene.camera) {
            const cameraEl = document.createElement('a-entity');
            cameraEl.setAttribute('camera', '');
            cameraEl.setAttribute('position', '0 1.6 3');
            cameraEl.setAttribute('look-controls', '');
            scene.appendChild(cameraEl);
          }
          
          lastHTML = scene.outerHTML;
          
          // Function to send update
          function sendUpdate() {
            if (pausePolling) return;
            const currentHTML = scene.outerHTML;
            if (currentHTML !== lastHTML) {
              console.log('Change detected! Sending to parent');
              lastHTML = currentHTML;
              window.parent.postMessage({
                type: 'aframe-scene-update',
                html: currentHTML
              }, '*');
            }
          }
          
          // Listen to inspector events
          scene.addEventListener('inspectortoggle', function(evt) {
            console.log('Inspector toggled:', evt.detail);
          });
          
          // Listen to componentchanged events on all entities
          scene.addEventListener('componentchanged', function(evt) {
            console.log('Component changed:', evt.detail.name, 'on', evt.target);
            
            // Sync the component data back to HTML attribute
            const entity = evt.target;
            const componentName = evt.detail.name;
            const component = entity.components[componentName];
            
            if (component && component.data) {
              // Special handling for material component with color
              if (componentName === 'material' && component.data.color) {
                entity.setAttribute('color', component.data.color);
                console.log('Synced color:', component.data.color);
              }
              
              // For other components, try to stringify and set
              try {
                if (typeof component.data === 'object') {
                  const stringified = AFRAME.utils.styleParser.stringify(component.data);
                  if (stringified) {
                    entity.setAttribute(componentName, stringified);
                  }
                } else {
                  entity.setAttribute(componentName, component.data);
                }
              } catch (e) {
                console.log('Could not sync component', componentName, e);
              }
            }
            
            setTimeout(sendUpdate, 100);
          }, true); // Use capture to get all child events
          
          // Use MutationObserver on the scene element
          const observer = new MutationObserver(function(mutations) {
            console.log('Mutation detected:', mutations.length, 'changes');
            sendUpdate();
          });
          
          observer.observe(scene, {
            attributes: true,
            attributeOldValue: true,
            childList: true,
            subtree: true,
            characterData: true
          });
          
          // Poll as backup
          setInterval(function() {
            sendUpdate();
          }, 2000);
        });
      }
    });
  </script>
</body>
</html>`;
        }

        // Function to update iframe from textarea
        function updateIframeFromTextarea() {
          const sceneHtml = $textarea.val() || '<a-scene embedded vr-mode-ui="enabled: false"></a-scene>';
          
          if (!iframe) {
            // Create iframe
            iframe = document.createElement('iframe');
            iframe.style.width = '100%';
            iframe.style.height = '100%';
            iframe.style.border = 'none';
            $preview.empty().append(iframe);
          }
          
          // Pause polling in iframe before updating
          if (iframe.contentWindow) {
            iframe.contentWindow.postMessage({ type: 'pause-polling' }, '*');
          }
          
          // Write HTML to iframe
          const doc = iframe.contentDocument || iframe.contentWindow.document;
          doc.open();
          doc.write(createIframeHTML(sceneHtml));
          doc.close();
        }

        // Listen for messages from iframe
        window.addEventListener('message', function(event) {
          console.log('Message received:', event.data);
          if (event.data && event.data.type === 'aframe-scene-update') {
            console.log('Received scene update from iframe');
            if (!isUpdatingFromIframe) {
              isUpdatingFromIframe = true;
              clearTimeout(updateTimeout);
              updateTimeout = setTimeout(function() {
                console.log('Applying update to textarea. Length:', event.data.html.length);
                $textarea.val(event.data.html);
                // Trigger change event so Drupal knows the field changed
                $textarea.trigger('change');
                console.log('Updated textarea from iframe:', event.data.html.substring(0, 100) + '...');
                isUpdatingFromIframe = false;
              }, 300);
            } else {
              console.log('Skipping update - already updating');
            }
          }
        });

        // Update iframe on textarea change (manual edits)
        $textarea.on('input', function() {
          clearTimeout(updateTimeout);
          updateTimeout = setTimeout(function() {
            if (!isUpdatingFromIframe) {
              updateIframeFromTextarea();
            }
          }, 1000);
        });

        // Initialize the iframe
        updateIframeFromTextarea();
        
        // Debug: Log textarea value on form submit
        $textarea.closest('form').on('submit', function() {
          console.log('Form submitting. Textarea value length:', $textarea.val().length);
          console.log('Textarea value preview:', $textarea.val().substring(0, 200));
        });
      });
    }
  };

})(jQuery, Drupal, drupalSettings, once);
