<?php

namespace Drupal\aframe\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure A-Frame settings.
 */
class AframeSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['aframe.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'aframe_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('aframe.settings');

    $form['library'] = [
      '#type' => 'details',
      '#title' => $this->t('Library Settings'),
      '#open' => TRUE,
    ];

    $form['library']['library_source'] = [
      '#type' => 'radios',
      '#title' => $this->t('Library Source'),
      '#options' => [
        'cdn' => $this->t('CDN (Content Delivery Network)'),
        'local' => $this->t('Local (self-hosted)'),
      ],
      '#default_value' => $config->get('library_source') ?? 'cdn',
      '#description' => $this->t('Choose whether to load A-Frame from CDN or a local copy. CDN is recommended for most sites.'),
    ];

    $form['library']['aframe_version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('A-Frame Version'),
      '#default_value' => $config->get('aframe_version') ?? '1.5.0',
      '#description' => $this->t('Specify the A-Frame version to use (e.g., 1.5.0).'),
      '#required' => TRUE,
    ];

    $form['scene_defaults'] = [
      '#type' => 'details',
      '#title' => $this->t('Default Scene Settings'),
      '#open' => TRUE,
    ];

    $form['scene_defaults']['background_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Background Color'),
      '#default_value' => $config->get('background_color') ?? '#ECECEC',
      '#description' => $this->t('Default background color for scenes (hex color code).'),
      '#size' => 10,
    ];

    $form['scene_defaults']['enable_fog'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Fog'),
      '#default_value' => $config->get('enable_fog') ?? FALSE,
    ];

    $form['scene_defaults']['fog_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Fog Type'),
      '#options' => [
        'linear' => $this->t('Linear'),
        'exponential' => $this->t('Exponential'),
      ],
      '#default_value' => $config->get('fog_type') ?? 'linear',
      '#states' => [
        'visible' => [
          ':input[name="enable_fog"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['scene_defaults']['fog_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Fog Color'),
      '#default_value' => $config->get('fog_color') ?? '#ECECEC',
      '#size' => 10,
      '#states' => [
        'visible' => [
          ':input[name="enable_fog"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['scene_defaults']['fog_near'] = [
      '#type' => 'number',
      '#title' => $this->t('Fog Near Distance'),
      '#default_value' => $config->get('fog_near') ?? 1,
      '#min' => 0,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="enable_fog"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['scene_defaults']['fog_far'] = [
      '#type' => 'number',
      '#title' => $this->t('Fog Far Distance'),
      '#default_value' => $config->get('fog_far') ?? 100,
      '#min' => 1,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="enable_fog"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['lighting'] = [
      '#type' => 'details',
      '#title' => $this->t('Default Lighting'),
      '#open' => FALSE,
    ];

    $form['lighting']['ambient_light_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ambient Light Color'),
      '#default_value' => $config->get('ambient_light_color') ?? '#FFFFFF',
      '#size' => 10,
    ];

    $form['lighting']['ambient_light_intensity'] = [
      '#type' => 'number',
      '#title' => $this->t('Ambient Light Intensity'),
      '#default_value' => $config->get('ambient_light_intensity') ?? 0.5,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
    ];

    $form['lighting']['directional_light_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Directional Light Color'),
      '#default_value' => $config->get('directional_light_color') ?? '#FFFFFF',
      '#size' => 10,
    ];

    $form['lighting']['directional_light_intensity'] = [
      '#type' => 'number',
      '#title' => $this->t('Directional Light Intensity'),
      '#default_value' => $config->get('directional_light_intensity') ?? 0.6,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
    ];

    $form['inspector'] = [
      '#type' => 'details',
      '#title' => $this->t('Inspector Settings'),
      '#open' => FALSE,
    ];

    $roles = user_role_names(TRUE);
    $form['inspector']['inspector_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles Allowed to Use Inspector'),
      '#options' => $roles,
      '#default_value' => $config->get('inspector_roles') ?? [],
      '#description' => $this->t('Select which roles can access the A-Frame visual inspector.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('aframe.settings')
      ->set('library_source', $form_state->getValue('library_source'))
      ->set('aframe_version', $form_state->getValue('aframe_version'))
      ->set('background_color', $form_state->getValue('background_color'))
      ->set('enable_fog', $form_state->getValue('enable_fog'))
      ->set('fog_type', $form_state->getValue('fog_type'))
      ->set('fog_color', $form_state->getValue('fog_color'))
      ->set('fog_near', $form_state->getValue('fog_near'))
      ->set('fog_far', $form_state->getValue('fog_far'))
      ->set('ambient_light_color', $form_state->getValue('ambient_light_color'))
      ->set('ambient_light_intensity', $form_state->getValue('ambient_light_intensity'))
      ->set('directional_light_color', $form_state->getValue('directional_light_color'))
      ->set('directional_light_intensity', $form_state->getValue('directional_light_intensity'))
      ->set('inspector_roles', array_filter($form_state->getValue('inspector_roles')))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
