<?php

namespace Drupal\aframe\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;

/**
 * Plugin implementation of the 'aframe_model' formatter.
 *
 * @FieldFormatter(
 *   id = "aframe_model",
 *   label = @Translation("A-Frame 3D Model Viewer"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class AframeModelFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'width' => '100%',
      'height' => '600px',
      'camera_position' => '0 1.6 3',
      'camera_rotation' => '0 0 0',
      'model_scale' => '1 1 1',
      'model_rotation' => '0 0 0',
      'model_position' => '0 0 0',
      'background_color' => '',
      'use_default_lighting' => 1,
      'custom_ambient_color' => '#FFFFFF',
      'custom_ambient_intensity' => '0.5',
      'custom_directional_color' => '#FFFFFF',
      'custom_directional_intensity' => '0.6',
      'auto_rotate' => 0,
      'show_download_button' => 1,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#description' => $this->t('Scene width (e.g., 100%, 800px).'),
    ];

    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Scene height (e.g., 600px, 50vh).'),
    ];

    $elements['camera_position'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Camera Position'),
      '#default_value' => $this->getSetting('camera_position'),
      '#description' => $this->t('Camera position in x y z format (e.g., 0 1.6 3).'),
    ];

    $elements['camera_rotation'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Camera Rotation'),
      '#default_value' => $this->getSetting('camera_rotation'),
      '#description' => $this->t('Camera rotation in degrees (e.g., 0 0 0).'),
    ];

    $elements['model_position'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Position'),
      '#default_value' => $this->getSetting('model_position'),
      '#description' => $this->t('Model position in x y z format (e.g., 0 0 0).'),
    ];

    $elements['model_scale'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Scale'),
      '#default_value' => $this->getSetting('model_scale'),
      '#description' => $this->t('Model scale in x y z format (e.g., 1 1 1).'),
    ];

    $elements['model_rotation'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Rotation'),
      '#default_value' => $this->getSetting('model_rotation'),
      '#description' => $this->t('Model rotation in degrees (e.g., 0 45 0).'),
    ];

    $elements['auto_rotate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-rotate model'),
      '#default_value' => $this->getSetting('auto_rotate'),
      '#description' => $this->t('Automatically rotate the model continuously.'),
    ];

    $elements['show_download_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show download button'),
      '#default_value' => $this->getSetting('show_download_button'),
      '#description' => $this->t('Display a button to download the 3D model file.'),
    ];

    $elements['background_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Background Color'),
      '#default_value' => $this->getSetting('background_color'),
      '#description' => $this->t('Background color (hex code). Leave empty to use global default.'),
      '#size' => 10,
    ];

    $elements['use_default_lighting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use default lighting settings'),
      '#default_value' => $this->getSetting('use_default_lighting'),
    ];

    $elements['custom_ambient_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Ambient Light Color'),
      '#default_value' => $this->getSetting('custom_ambient_color'),
      '#size' => 10,
      '#states' => [
        'visible' => [
          ':input[name*="use_default_lighting"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $elements['custom_ambient_intensity'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom Ambient Light Intensity'),
      '#default_value' => $this->getSetting('custom_ambient_intensity'),
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
      '#states' => [
        'visible' => [
          ':input[name*="use_default_lighting"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $elements['custom_directional_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Directional Light Color'),
      '#default_value' => $this->getSetting('custom_directional_color'),
      '#size' => 10,
      '#states' => [
        'visible' => [
          ':input[name*="use_default_lighting"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $elements['custom_directional_intensity'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom Directional Light Intensity'),
      '#default_value' => $this->getSetting('custom_directional_intensity'),
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
      '#states' => [
        'visible' => [
          ':input[name*="use_default_lighting"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    $summary[] = $this->t('Dimensions: @width × @height', [
      '@width' => $this->getSetting('width'),
      '@height' => $this->getSetting('height'),
    ]);
    
    $summary[] = $this->t('Camera: @pos', [
      '@pos' => $this->getSetting('camera_position'),
    ]);

    if ($this->getSetting('auto_rotate')) {
      $summary[] = $this->t('Auto-rotate enabled');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $config = \Drupal::config('aframe.settings');

    foreach ($items as $delta => $item) {
      // Skip if no target_id or entity
      if (empty($item->target_id) || empty($item->entity)) {
        continue;
      }

      /** @var \Drupal\file\FileInterface $file */
      $file = $item->entity;
      
      if (!$file || !$file->getFileUri()) {
        continue;
      }

      // Ensure file has MIME type set
      if (!$file->getMimeType()) {
        $file->setMimeType(\Drupal::service('file.mime_type.guesser')->guessMimeType($file->getFileUri()));
      }

      // Generate absolute URL for A-Frame to load the file
      $file_url = $file->createFileUrl(FALSE);
      
      $elements[$delta] = $this->buildViewElement($file_url, $config);
    }

    return $elements;
  }

  /**
   * Build a view element for a 3D model.
   *
   * @param string $file_url
   *   The URL to the model file.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return array
   *   A render array.
   */
  protected function buildViewElement($file_url, $config) {
    // Determine background color
    $background_color = $this->getSetting('background_color');
    if (empty($background_color)) {
      $background_color = $config->get('background_color') ?? '#ECECEC';
    }

    // Determine lighting
    if ($this->getSetting('use_default_lighting')) {
      $ambient_color = $config->get('ambient_light_color') ?? '#FFFFFF';
      $ambient_intensity = $config->get('ambient_light_intensity') ?? 0.5;
      $directional_color = $config->get('directional_light_color') ?? '#FFFFFF';
      $directional_intensity = $config->get('directional_light_intensity') ?? 0.6;
    }
    else {
      $ambient_color = $this->getSetting('custom_ambient_color');
      $ambient_intensity = $this->getSetting('custom_ambient_intensity');
      $directional_color = $this->getSetting('custom_directional_color');
      $directional_intensity = $this->getSetting('custom_directional_intensity');
    }

    // Determine libraries to load
    $libraries = [
      'aframe/aframe',
      'aframe/model-viewer',
    ];

    return [
      '#theme' => 'aframe_model_viewer',
      '#file_url' => $file_url,
      '#width' => $this->getSetting('width'),
      '#height' => $this->getSetting('height'),
      '#camera_position' => $this->getSetting('camera_position'),
      '#camera_rotation' => $this->getSetting('camera_rotation'),
      '#model_position' => $this->getSetting('model_position'),
      '#model_scale' => $this->getSetting('model_scale'),
      '#model_rotation' => $this->getSetting('model_rotation'),
      '#background_color' => $background_color,
      '#ambient_color' => $ambient_color,
      '#ambient_intensity' => $ambient_intensity,
      '#directional_color' => $directional_color,
      '#directional_intensity' => $directional_intensity,
      '#auto_rotate' => $this->getSetting('auto_rotate'),
      '#show_download_button' => $this->getSetting('show_download_button'),
      '#attached' => [
        'library' => $libraries,
      ],
    ];
  }

}
