<?php

namespace Drupal\aframe\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'aframe_scene' formatter.
 *
 * @FieldFormatter(
 *   id = "aframe_scene",
 *   label = @Translation("A-Frame Scene Viewer"),
 *   field_types = {
 *     "text_long",
 *     "string_long"
 *   }
 * )
 */
class AframeSceneFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'width' => '100%',
      'height' => '600px',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#description' => $this->t('Scene width (e.g., 100%, 800px).'),
    ];

    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Scene height (e.g., 600px, 50vh).'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    $summary[] = $this->t('Dimensions: @width × @height', [
      '@width' => $this->getSetting('width'),
      '@height' => $this->getSetting('height'),
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $scene_html = $item->value;
      
      if (empty($scene_html)) {
        continue;
      }

      // Create unique ID for this scene
      $scene_id = 'aframe-scene-' . $delta . '-' . substr(md5($scene_html), 0, 8);

      // Detect which libraries are needed based on scene content
      $aframe_version = '1.7.1';
      $scripts = '<script src="https://cdn.jsdelivr.net/npm/aframe@' . $aframe_version . '/dist/aframe-master.min.js"></script>';
      
      // Check if scene uses orbit-controls
      if (stripos($scene_html, 'orbit-controls') !== FALSE) {
        $scripts .= '<script src="https://cdn.jsdelivr.net/npm/aframe-orbit-controls@1.3.0/dist/aframe-orbit-controls.min.js"></script>';
      }

      // Create iframe HTML
      $iframe_html = '<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>A-Frame Scene</title>
  ' . $scripts . '
  <style>
    html, body { margin: 0; padding: 0; width: 100%; height: 100%; overflow: hidden; }
    a-scene { width: 100%; height: 100%; }
  </style>
</head>
<body>
  ' . $scene_html . '
</body>
</html>';

      $elements[$delta] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['aframe-scene-viewer'],
          'id' => $scene_id,
        ],
        'iframe' => [
          '#type' => 'html_tag',
          '#tag' => 'iframe',
          '#attributes' => [
            'srcdoc' => $iframe_html,
            'style' => sprintf('width: %s; height: %s; border: none; display: block;', 
              $this->getSetting('width'), 
              $this->getSetting('height')
            ),
            'allowfullscreen' => TRUE,
          ],
        ],
      ];
    }

    return $elements;
  }

}
