<?php

namespace Drupal\aframe\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'aframe_inspector' widget.
 *
 * @FieldWidget(
 *   id = "aframe_inspector",
 *   label = @Translation("A-Frame Inspector Scene Builder"),
 *   field_types = {
 *     "text_long",
 *     "string_long"
 *   }
 * )
 */
class AframeInspectorWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'include_base_scene' => TRUE,
      'default_camera_position' => '0 1.6 3',
      'enable_stats' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['include_base_scene'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include base scene structure'),
      '#default_value' => $this->getSetting('include_base_scene'),
      '#description' => $this->t('Automatically add basic scene, camera, and lighting when creating new content.'),
    ];

    $elements['default_camera_position'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Camera Position'),
      '#default_value' => $this->getSetting('default_camera_position'),
      '#description' => $this->t('Default camera position for new scenes (e.g., 0 1.6 3).'),
    ];

    $elements['enable_stats'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable performance stats'),
      '#default_value' => $this->getSetting('enable_stats'),
      '#description' => $this->t('Show FPS and performance statistics in the scene.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    if ($this->getSetting('include_base_scene')) {
      $summary[] = $this->t('Base scene structure included');
    }

    if ($this->getSetting('enable_stats')) {
      $summary[] = $this->t('Stats enabled');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $config = \Drupal::config('aframe.settings');
    $current_user = \Drupal::currentUser();
    
    // Check if user has permission to use inspector
    $inspector_roles = $config->get('inspector_roles') ?? [];
    $user_roles = $current_user->getRoles();
    $can_use_inspector = !empty(array_intersect($user_roles, $inspector_roles)) || $current_user->hasPermission('administer site configuration');

    // Get the current value from the field
    $value = isset($items[$delta]->value) ? $items[$delta]->value : '';
    
    // If empty and base scene is enabled, provide a starter template
    // Only do this for truly new/empty fields, not on every edit
    if (empty($value) && $this->getSetting('include_base_scene') && $items->getEntity()->isNew()) {
      $background_color = $config->get('background_color') ?? '#ECECEC';
      $ambient_color = $config->get('ambient_light_color') ?? '#FFFFFF';
      $ambient_intensity = $config->get('ambient_light_intensity') ?? 0.5;
      $directional_color = $config->get('directional_light_color') ?? '#FFFFFF';
      $directional_intensity = $config->get('directional_light_intensity') ?? 0.6;
      $camera_position = $this->getSetting('default_camera_position');
      
      $stats = $this->getSetting('enable_stats') ? ' stats' : '';
      
      $value = <<<HTML
<a-scene background="color: {$background_color}"{$stats}>
  <a-entity id="camera" camera position="{$camera_position}" look-controls></a-entity>
  <a-entity light="type: ambient; color: {$ambient_color}; intensity: {$ambient_intensity}"></a-entity>
  <a-entity light="type: directional; color: {$directional_color}; intensity: {$directional_intensity}" position="1 1 1"></a-entity>
  
  <!-- Add your entities here -->
  <a-box position="0 0.5 -3" rotation="0 45 0" color="#4CC3D9"></a-box>
  <a-sphere position="1 1.25 -5" radius="1.25" color="#EF2D5E"></a-sphere>
  <a-cylinder position="-1 0.75 -3" radius="0.5" height="1.5" color="#FFC65D"></a-cylinder>
  <a-plane position="0 0 -4" rotation="-90 0 0" width="4" height="4" color="#7BC8A4"></a-plane>
</a-scene>
HTML;
    }

    // Add wrapper and preview as prefix to the value element
    if ($can_use_inspector) {
      $element['#prefix'] = '<div class="aframe-widget-wrapper"><div class="aframe-scene-preview"></div><div class="description">' . 
        $this->t('Use the visual inspector: Press <strong>Ctrl+Alt+I</strong> (or <strong>Cmd+Option+I</strong> on Mac) to open it. Changes are automatically saved.') . 
        '</div>';
      $element['#suffix'] = '</div>';
      $element['#attached']['library'][] = 'aframe/inspector-widget';
    }
    
    // Wrap textarea in collapsible details for advanced users
    $element['value'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced: HTML Source'),
      '#open' => FALSE,
      'value' => [
        '#type' => 'textarea',
        '#disabled' => TRUE,
        '#default_value' => $value,
        '#rows' => 15,
        '#description' => $this->t('The scene HTML source code. This is automatically updated when you use the visual inspector.'),
        '#attributes' => [
          'class' => ['aframe-scene-textarea'],
          'data-aframe-widget' => 'true',
        ],
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // The details element creates nested structure: $values[0]['value']['value']
    // We need to flatten it to: $values[0]['value']
    foreach ($values as $delta => &$item) {
      if (isset($item['value']) && is_array($item['value']) && isset($item['value']['value'])) {
        $item['value'] = $item['value']['value'];
      }
    }
    return $values;
  }

}
