<?php

namespace Drupal\aframe\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Plugin\Field\FieldWidget\FileWidget;

/**
 * Plugin implementation of the 'aframe_model_widget' widget.
 *
 * @FieldWidget(
 *   id = "aframe_model_widget",
 *   label = @Translation("A-Frame 3D Model Upload with Preview"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class AframeModelWidget extends FileWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'preview_width' => '100%',
      'preview_height' => '400px',
      'show_preview' => TRUE,
      'camera_position' => '0 1.6 3',
      'model_scale' => '1 1 1',
      'auto_rotate' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['show_preview'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show 3D model preview'),
      '#default_value' => $this->getSetting('show_preview'),
      '#description' => $this->t('Display a preview of the uploaded 3D model below the upload field.'),
    ];

    $elements['preview_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Preview Width'),
      '#default_value' => $this->getSetting('preview_width'),
      '#description' => $this->t('Preview width (e.g., 100%, 600px).'),
      '#states' => [
        'visible' => [
          ':input[name*="show_preview"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $elements['preview_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Preview Height'),
      '#default_value' => $this->getSetting('preview_height'),
      '#description' => $this->t('Preview height (e.g., 400px, 30vh).'),
      '#states' => [
        'visible' => [
          ':input[name*="show_preview"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $elements['camera_position'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Camera Position'),
      '#default_value' => $this->getSetting('camera_position'),
      '#description' => $this->t('Camera position in x y z format (e.g., 0 1.6 3).'),
      '#states' => [
        'visible' => [
          ':input[name*="show_preview"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $elements['model_scale'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Scale'),
      '#default_value' => $this->getSetting('model_scale'),
      '#description' => $this->t('Model scale in x y z format (e.g., 1 1 1).'),
      '#states' => [
        'visible' => [
          ':input[name*="show_preview"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $elements['auto_rotate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-rotate preview'),
      '#default_value' => $this->getSetting('auto_rotate'),
      '#description' => $this->t('Automatically rotate the model in the preview.'),
      '#states' => [
        'visible' => [
          ':input[name*="show_preview"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    if ($this->getSetting('show_preview')) {
      $summary[] = $this->t('Preview: @width × @height', [
        '@width' => $this->getSetting('preview_width'),
        '@height' => $this->getSetting('preview_height'),
      ]);

      if ($this->getSetting('auto_rotate')) {
        $summary[] = $this->t('Auto-rotate enabled');
      }
    }
    else {
      $summary[] = $this->t('No preview');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Add preview if setting is enabled and file exists
    if ($this->getSetting('show_preview') && !empty($items[$delta]->target_id)) {
      /** @var \Drupal\file\FileInterface $file */
      $file = $items[$delta]->entity;
      
      if ($file && $file->getFileUri()) {
        $file_url = $file->createFileUrl(FALSE);
        $config = \Drupal::config('aframe.settings');

        // Get background color from config
        $background_color = $config->get('background_color') ?? '#ECECEC';
        
        // Get lighting settings from config
        $ambient_color = $config->get('ambient_light_color') ?? '#FFFFFF';
        $ambient_intensity = $config->get('ambient_light_intensity') ?? 0.5;
        $directional_color = $config->get('directional_light_color') ?? '#FFFFFF';
        $directional_intensity = $config->get('directional_light_intensity') ?? 0.6;

        // Add preview below the file field
        $element['preview'] = [
          '#type' => 'container',
          '#weight' => 100,
          '#attributes' => [
            'class' => ['aframe-widget-preview'],
            'style' => 'width: 400px; display: block; margin-top: 10px;',
          ],
          'viewer' => [
            '#theme' => 'aframe_model_viewer',
            '#file_url' => $file_url,
            '#width' => $this->getSetting('preview_width'),
            '#height' => $this->getSetting('preview_height'),
            '#camera_position' => $this->getSetting('camera_position'),
            '#camera_rotation' => '0 0 0',
            '#model_position' => '0 0 0',
            '#model_scale' => $this->getSetting('model_scale'),
            '#model_rotation' => '0 0 0',
            '#background_color' => $background_color,
            '#ambient_color' => $ambient_color,
            '#ambient_intensity' => $ambient_intensity,
            '#directional_color' => $directional_color,
            '#directional_intensity' => $directional_intensity,
            '#auto_rotate' => $this->getSetting('auto_rotate'),
            '#show_download_button' => FALSE,
            '#attached' => [
              'library' => [
                'aframe/aframe',
                'aframe/model-viewer',
              ],
            ],
          ],
        ];
      }
    }

    return $element;
  }

}
