<?php

namespace Drupal\afterburner_queue\Commands;

use Drupal\afterburner_queue\Task\QueueItemTask;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Queue\RequeueException;
use Drupal\Core\Queue\SuspendQueueException;
use Drush\Commands\DrushCommands;
use Spatie\Async\Pool;

class QueueCommands extends DrushCommands {

  /**
   * @var \Drupal\Core\Queue\QueueWorkerManagerInterface
   */
  private QueueWorkerManagerInterface $queueWorkerManager;

  /**
   * @var \Drupal\Core\Queue\QueueFactory
   */
  private QueueFactory $queueFactory;

  /**
   * @param \Drupal\Core\Queue\QueueWorkerManagerInterface $queueWorkerManager
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   */
  public function __construct(QueueWorkerManagerInterface $queueWorkerManager, QueueFactory $queueFactory) {
    parent::__construct();

    $this->queueWorkerManager = $queueWorkerManager;
    $this->queueFactory = $queueFactory;
  }

  /**
   * @command queue:run-async
   * @param string $name The name of the queue to run, as defined in either hook_queue_info or hook_cron_queue_info.
   * @validate-queue name
   * @option items-limit The maximum number of items allowed to run the queue.
   * @option lease-time The maximum number of seconds that an item remains claimed.
   * @option concurrency The maximum number of processes to spawn.
   * @option timeout How long the child process can run.
   * @option sleep-time Interval between child process checks.
   */
  public function run($name, $options = ['items-limit' => self::REQ, 'lease-time' => self::REQ, 'concurrency' => self::REQ, 'timeout' => self::REQ, 'sleep-time' => self::REQ]) {
    $start = microtime(true);

    $items_limit = (int) $options['items-limit'];
    $concurrency = (int) ($options['concurrency'] ?? 20);
    $timeout = (int) ($options['timeout'] ?? 300);
    $sleepTime = (int) ($options['sleep-time'] ?? 50000);

    $info = $this->queueWorkerManager->getDefinition($name);
    $queue = $this->queueFactory->get($name);
    $claimed = 0;
    $processed = 0;
    $lease_time = $options['lease-time'] ?? $info['cron']['time'] ?? 30;

    $pool = (Pool::create())
      ->concurrency($concurrency)
      ->timeout($timeout)
      ->sleepTime($sleepTime);

    // @TODO Do not claim all items at once, use a generator.
    while ((!$items_limit || $claimed < $items_limit) && ($item = $queue->claimItem($lease_time))) {
      $pool->add(new QueueItemTask($name, $item))
        ->then(function ($output) use ($name, $queue, $item, &$processed) {
          $this->logger()->success(dt('Processed item @id from @name queue.', ['@name' => $name, '@id' => $item->item_id]));
          $processed++;
        })->catch(function (RequeueException $e) use ($queue, $item) {
          $this->logger()->error($e->getMessage());
        })->catch(function (SuspendQueueException $e) use ($queue, $item) {
          $this->logger()->error($e->getMessage());
          throw new \Exception($e->getMessage());
        })->catch(function (\Exception $e) {
          $this->logger()->error($e->getMessage());
        });

      // @TODO item-limits works differently, core only counts processed queue items.
      $claimed++;
    }

    $pool->wait();

    $elapsed = microtime(true) - $start;
    $this->logger()->success(dt('Processed @processed items from the @name queue in @elapsed sec.', [
      '@processed' => $processed,
      '@name' => $name,
      '@elapsed' => $elapsed,
    ]));
  }

}
