<?php

namespace Drupal\aggrid;

use Drupal\aggrid\Entity\Aggrid;

/**
 * Helper functions for agGrid config Entities
 *
 */
class AggridConfigHelpers {

  public function getDefaults($aggrid_id) {
    $aggridDefault = [];

    // Get config for aggrid.
    $config = \Drupal::config('aggrid.general');
    // Set the aggrid setting variable
    $aggridgsjson = json_decode($config->get('aggridgsjson'));
    // Get the global aggrid row settings
    if(isset($aggridgsjson->rowSettings)) {
      $gsRowSettings = json_decode(json_encode($aggridgsjson->rowSettings));
    }

    // Fetch aggrid entity
    $aggridEntity = Aggrid::load($aggrid_id);

    if (!empty($aggridEntity)) {
      $aggridDefault['default'] = json_decode($aggridEntity->get('aggridDefault'));

      $aggridDefault['addOptions'] = @json_decode($aggridEntity->get('addOptions'));

      $aggridRowSettings = @json_decode(json_encode($aggridDefault['default']->rowSettings), true);

      // Ensure $gsRowSettings is an array before using array_replace_recursive
      if (isset($gsRowSettings)) {
          $gsRowSettings = (array) $gsRowSettings; // Cast to array
      }

      if (isset($gsRowSettings) && isset($aggridRowSettings)) {
        $aggridRowSettings = array_replace_recursive($gsRowSettings, $aggridRowSettings);
      }
      $aggridDefault['aggridRowSettings'] = $aggridRowSettings;
    }

    return $aggridDefault;
  }

  public function getHeaders($columnDefs, $dataHeaderSettings = null) {

    $getHeaders = [];

    $columns = [];
    $headers = [];
    $headersNameFull = [];

    // Build table.
    $rowIndex = 0;
    $colIndex = 0;

    foreach ($columnDefs as $column) {
      $rowIndex = $rowIndex > 1 ? $rowIndex : 1;
      $colIndex++;

      $columns[1][$colIndex][$colIndex] = [];
      $columns[1][$colIndex][$colIndex]['headerName'] = $column->headerName;
      $columns[1][$colIndex][$colIndex]['headerNameFull'] = $column->headerName;
      $columns[1][$colIndex][$colIndex]['field'] = isset($column->field) ? $column->field : NULL;
      $columns[1][$colIndex][$colIndex]['width'] = isset($column->width) ? $column->width : NULL;
      $columns[1][$colIndex][$colIndex]['minWidth'] = isset($column->minWidth) ? $column->minWidth : NULL;
      $columns[1][$colIndex][$colIndex]['viewHide'] = isset($column->viewHide) ? $column->viewHide : false;

      // Set flat header full name
      if (!empty($column->field)) {
        $headersNameFull[$column->field] = $column->headerName;
      }


      // If children, then dive down for headers, otherwise establish column.
      if (isset($column->children)) {
        $colCount = count($column->children);
        $columns[1][$colIndex][$colIndex]['colspan'] = $colCount;
        $count2 = 0;
        foreach ($column->children as $child) {
          // If column is hidden on view, reduce previous colspan by one
          if (isset($child->viewHide)
            && $child->viewHide) {
            $colCount = $columns[1][$colIndex][$colIndex]['colspan'];
            $columns[1][$colIndex][$colIndex]['colspan'] = $colCount - 1;
          }

          $rowIndex = $rowIndex > 2 ? $rowIndex : 2;
          $count2++;

          $columns[2][$colIndex][$count2] = [];
          $columns[2][$colIndex][$count2]['headerName'] = $child->headerName;
          $columns[2][$colIndex][$count2]['headerNameFull'] = $column->headerName . ' - ' . $child->headerName;
          $columns[2][$colIndex][$count2]['field'] = isset($child->field) ? $child->field : NULL;
          $columns[2][$colIndex][$count2]['width'] = isset($child->width) ? $child->width : NULL;
          $columns[2][$colIndex][$count2]['minWidth'] = isset($child->minWidth) ? $child->minWidth : NULL;
          $columns[2][$colIndex][$count2]['viewHide'] = isset($child->viewHide) ? $child->viewHide : false;
          // Set flat header full name
          if (!empty($child->field)){
            $headersNameFull[$child->field] = $column->headerName . ' - ' . $child->headerName;
          }

          // if the viewhide is true viewHide as true for this child
          if ($columns[1][$colIndex][$colIndex]['viewHide']) {
            $columns[2][$colIndex][$count2]['viewHide'] = true;
          }

          // Set the field if available.
          if (isset($child->field)) {
            $headers[] = $child->field;
            $columns[0][$child->field] = [];
            $columns[0][$child->field]['viewHide'] = $columns[2][$colIndex][$count2]['viewHide'];
          }

          // Just one for colspan.
          $columns[2][$colIndex][$count2]['colspan'] = 1;
        }

      }
      else {
        // Just one for colspan.
        $columns[1][$colIndex][$colIndex]['colspan'] = 1;
      }

      // If the parent column colspan is now 0, then default viewHide to true (all children are hidden)
      if ($columns[1][$colIndex][$colIndex]['colspan'] <= 0) {
        $columns[1][$colIndex][$colIndex]['viewHide'] = true;
      }

      // If no children, set the field if available.
      if (!isset($column->children) && isset($column->field)) {
        $headers[] = $column->field;
        $columns[0][$column->field] = [];
        $columns[0][$column->field]['viewHide'] = $columns[1][$colIndex][$colIndex]['viewHide'];
      }
    }

    // Set return
    $getHeaders['rowIndex'] = $rowIndex;
    $getHeaders['colIndex'] = $colIndex;
    $getHeaders['headers'] = $headers;
    $getHeaders['headersNameFull'] = $headersNameFull;
    $getHeaders['columns'] = $columns;

    return $getHeaders;
  }

  public function getRowSettings($aggridRowSettings, $headers, $rowData, $rowPrefix) {
    // Set global complimentary switch
    // Set the rest of the row settings
    $rowSettings[][] = '';
    if (is_array($rowData)) {
      for ($i = 0; $i < count($rowData); $i++) {
        foreach ($headers as $field) {
          $rowSettings[$i][$field] = [];
          if (isset($aggridRowSettings[$rowPrefix . $i][$field])) {
            $rowSettings[$i][$field] = $rowSettings[$i][$field] + $aggridRowSettings[$rowPrefix . $i][$field];
          }
          if (isset($aggridRowSettings[$rowPrefix . $i]['rowDefault'])) {
            $rowSettings[$i][$field] = $rowSettings[$i][$field] + $aggridRowSettings[$rowPrefix . $i]['rowDefault'];
          }
          if (isset($aggridRowSettings[$rowPrefix . 'default'][$field])) {
            $rowSettings[$i][$field] = $rowSettings[$i][$field] + $aggridRowSettings[$rowPrefix . 'default'][$field];
          }
          if (isset($aggridRowSettings[$rowPrefix . 'default']['rowDefault'])) {
            $rowSettings[$i][$field] = $rowSettings[$i][$field] + $aggridRowSettings[$rowPrefix . 'default']['rowDefault'];
          }
        }
      }
    }

    return $rowSettings;
  }

  /**
   * Generates a map of row 1-based index (matching validation warning ID)
   * to their human-readable labels, concatenating multiple label columns.
   *
   * @param array $rowData
   * The row data array from the AgGrid config. (Expected to be 0-indexed.)
   * @param array $aggridRowSettings
   * The row settings array to determine which fields hold the label.
   *
   * @return array
   * An associative array where keys are 1-based row indices and values are the concatenated row label string.
   */
  public function getRowLabels(array $rowData, array $aggridRowSettings) {
    $rowLabels = [];
    $labelFields = [];

    // Collect all fields marked with "isRowLabel": true.
    if (isset($aggridRowSettings['default'])) {
        foreach ($aggridRowSettings['default'] as $field => $settings) {
            if (isset($settings['isRowLabel']) && $settings['isRowLabel'] === true) {
                $labelFields[] = $field;
            }
        }
    }

    // Fallback if no fields were explicitly marked. We fall back to 'id' if available.
    if (empty($labelFields) && !empty($rowData) && isset($rowData[0]['id'])) {
        $labelFields[] = 'id';
    }

    // Build the map: [1-based index => concatenated_row_label]
    foreach ($rowData as $i => $row) {
        // $i is the 0-based array index (0, 1, 2, ...).
        // The validation ID is 1-based (1, 2, 3, ...), so we use $i + 1 as the key.
        $row_id_key = $i;

        $label_parts = [];

        // Loop through all determined label fields and collect their values.
        foreach ($labelFields as $field) {
            if (isset($row[$field])) {
                // Ensure the value is not empty or just whitespace before adding.
                $value = trim((string) $row[$field]);
                if (!empty($value)) {
                    $label_parts[] = $value;
                }
            }
        }

        // Concatenate the parts with ' - '.
        if (!empty($label_parts)) {
            $rowLabels[$row_id_key] = implode(' - ', $label_parts);
        } else {
            // Fallback: If no label fields had data, use the generic row ID.
            $rowLabels[$row_id_key] = t('Row @id', ['@id' => $row_id_key]);
        }
    }

    return $rowLabels;
  }
}
