<?php

namespace Drupal\aggrid\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\aggrid\AggridConfigHelpers;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'aggrid_prevwarninglist_type' formatter.
 *
 * @FieldFormatter(
 * id = "aggrid_prevwarninglist_type",
 * label = @Translation("ag-Grid previous warning view mode"),
 * field_types = {
 * "aggrid"
 * }
 * )
 */
class AggridPrevwarninglistFormatterType extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The AgGrid configuration helpers.
   *
   * @var \Drupal\aggrid\AggridConfigHelpers
   */
  protected $aggridHelpers;

  /**
   * Constructs an AggridPrevwarninglistFormatterType object.
   *
   * @param string $plugin_id
   * The plugin_id for the formatter.
   * @param mixed $plugin_definition
   * The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   * The field definition.
   * @param array $settings
   * The formatter settings.
   * @param string $label
   * The formatter label.
   * @param string $view_mode
   * The view mode.
   * @param array $third_party_settings
   * Third party settings.
   * @param \Drupal\aggrid\AggridConfigHelpers $aggrid_helpers
   * The AgGrid configuration helpers.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, array $settings, $label, $view_mode, array $third_party_settings, AggridConfigHelpers $aggrid_helpers) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->aggridHelpers = $aggrid_helpers;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('aggrid.config_helpers')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_row_key' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);

    $element['show_row_key'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include AgGrid Row Key in summary'),
      '#default_value' => $this->getSetting('show_row_key'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = $this->t('Renders validation warnings as a collapsible list.');

    if ($this->getSetting('show_row_key')) {
      $summary[] = $this->t('Row key is included in warning summaries.');
    }

    return $summary;
  }

  /**
   * Formats a numeric value based on whether it is currency or a standard decimal.
   *
   * If currency, always formats to 2 decimal places.
   * If non-currency, formats to 0 decimals if the value is an integer, or 2 decimals otherwise.
   *
   * @param float $value
   * The numeric value to format.
   * @param bool $isCurrency
   * Whether to apply currency formatting (e.g., $1,000.00).
   *
   * @return string
   * The formatted string.
   */
  private function formatNumericValue(float $value, bool $isCurrency): string {

    // Determine the number of decimal places needed.
    if ($isCurrency) {
      // Currency should always be displayed with 2 decimal places.
      $decimals = 2;
    } else {
      // For non-currency, check if the value is effectively an integer
      // by comparing it to its rounded value (using a small epsilon for safety).
      if (abs($value - round($value)) < 0.00001) {
        $decimals = 0; // Display whole numbers without .00
      } else {
        $decimals = 2; // Display decimals with 2 places (e.g., 100.50)
      }
    }

    $formatted = number_format($value, $decimals, '.', ',');

    if ($isCurrency) {
      // Prepend the currency symbol.
      return '$' . $formatted;
    }

    // Return standard numeric formatting.
    return $formatted;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $all_warnings = [];

      // 1. Get the raw JSON string and the aggrid_id.
      $raw_json = $item->get('prevwarninglist')->getValue();
      $aggrid_id = $item->get('aggrid_id')->getValue();

      if (empty($raw_json) || !is_string($raw_json) || empty($aggrid_id)) {
        continue;
      }

      $decoded_data = json_decode($raw_json, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded_data)) {
        continue;
      }

      // 2. Fetch AgGrid Configuration for Labels
      $aggrid_config = $this->aggridHelpers->getDefaults($aggrid_id);

      $columnDefs = $aggrid_config['default']->columnDefs ?? [];
      $header_data = $this->aggridHelpers->getHeaders($columnDefs);
      $column_labels = $header_data['headersNameFull'] ?? [];

      $aggridRowSettings = $aggrid_config['aggridRowSettings'] ?? [];
      $rowData = $aggrid_config['default']->rowData ?? [];
      $rowDataArray = json_decode(json_encode($rowData), TRUE) ?? [];
      $row_labels_map = $this->aggridHelpers->getRowLabels($rowDataArray, $aggridRowSettings);

      // 3. Process the decoded data into a structured array for the Twig template.
      foreach ($decoded_data as $row_id => $validations) {

        if (!is_array($validations) || empty($validations)) { continue; }

        // Get the human-readable row label (1-based index).
        // Note: We use the 1-based row_id directly from the warning list keys.
        $row_label = $row_labels_map[$row_id] ?? $this->t('Row @id', ['@id' => $row_id]);

        foreach ($validations as $field_key => $field_data_array) {

          if (!is_array($field_data_array) || empty($field_data_array)) { continue; }

          // Use the mapped header name, fall back to the raw field key if not found.
          $column_label = $column_labels[$field_key] ?? $field_key;

          if (isset($field_data_array['prevwarning']) && is_array($field_data_array['prevwarning'])) {
            $warning = $field_data_array['prevwarning'];

            $rawCurValue = $warning['curValue'] ?? '0';
            $rawPrevValue = $warning['prevValue'] ?? '0';

            // Detect if it's currency based on the presence of '$' in the raw value.
            $is_currency = str_contains($rawCurValue, '$') || str_contains($rawPrevValue, '$');

            // Sanitize values by removing currency symbols ($) and commas (,) and casting to float for CALCULATION.
            $curValueSanitized = str_replace(['$', ','], '', $rawCurValue);
            $prevValueSanitized = str_replace(['$', ','], '', $rawPrevValue);

            // Cast to float to handle decimals (e.g., .07).
            $curValue = (float) $curValueSanitized;
            $prevValue = (float) $prevValueSanitized;

            // Calculation and type determination.
            $change = $curValue - $prevValue;

            if ($change > 0) {
              $highlow = 'HIGHER';
            } elseif (abs($change) < 0.00001) { // Check for near-zero change for floats
              $highlow = 'ZERO';
            } else {
              $highlow = 'LOWER';
            }

            // --- CORRECTED PERCENTAGE CALCULATION ---
            $raw_percent = 0.00;
            if (abs($prevValue) < 0.00001) {
              // If the base is near zero.
              if (abs($curValue) > 0.00001) {
                // Change from 0 to a value is mathematically infinite, but
                // for display purposes, we calculate the percentage based on the new value's magnitude.
                // Using 100% per unit change, or a proxy like $curValue * 100.
                // Since this happens for small $prevValue, using $curValue * 100 provides a large, clear warning.
                $raw_percent = abs($curValue) * 100;
              }
            } else {
              // Standard percentage change calculation: (New - Old) / Old * 100
              $raw_percent = ($change / $prevValue) * 100;
            }

            // Format the absolute value of the raw percentage to 2 decimal places.
            $percent = number_format(abs(round($raw_percent, 2)), 2);
            // ----------------------------------------

            // Apply smart formatting to the output values.
            $formatted_curValue = $this->formatNumericValue($curValue, $is_currency);
            $formatted_prevValue = $this->formatNumericValue($prevValue, $is_currency);
            $formatted_change = $this->formatNumericValue($change, $is_currency);

            $all_warnings[] = [
              'row_label' => $row_label,
              'column_label' => $column_label,
              'highlow' => $highlow,
              'percent' => $percent, // Now contains the correct percentage change
              'change' => $formatted_change,
              'curValue' => $formatted_curValue,
              'prevValue' => $formatted_prevValue,
              'message' => $warning['message'] ?? $this->t('N/A'),
            ];
          }
        }
      }

      // 4. Pass the structured data to the dedicated Twig template.
      if (!empty($all_warnings)) {
        $elements[$delta] = [
          '#theme' => 'aggrid_prevwarninglist_type',
          '#warnings' => $all_warnings,
          '#settings' => $this->getSettings(),
          '#attached' => [
            // Ensure the library is defined in aggrid.libraries.yml.
            'library' => ['aggrid/validation_formatter'],
          ],
        ];
      }
    }

    return $elements;
  }
}
