/**
 * Example: Registering custom tool renderers for AG-UI Chat.
 *
 * This file demonstrates how external modules can register their own
 * tool UI renderers to display custom components during tool execution.
 *
 * Include this file after the agui chat library loads, or use the
 * 'agui:tools:ready' event to ensure the API is available.
 */

// Method 1: Queue registrations before the API is ready
// This works even if your script loads before agui chat
window.AguiToolsQueue = window.AguiToolsQueue || [];
window.AguiToolsQueue.push({
  name: 'get_weather',
  config: {
    render: ({ args, status, result, container }) => {
      const city = args.city || 'Unknown';

      if (status === 'pending' || status === 'streaming') {
        container.innerHTML = `
          <div class="weather-card weather-card--loading">
            <div class="weather-card__icon">🌤️</div>
            <div class="weather-card__content">
              <div class="weather-card__city">${city}</div>
              <div class="weather-card__loading">Fetching weather...</div>
            </div>
          </div>
        `;
        return;
      }

      if (status === 'error') {
        container.innerHTML = `
          <div class="weather-card weather-card--error">
            <div class="weather-card__icon">⚠️</div>
            <div class="weather-card__content">
              <div class="weather-card__city">${city}</div>
              <div class="weather-card__error">Failed to fetch weather</div>
            </div>
          </div>
        `;
        return;
      }

      // Parse result (could be JSON string or object)
      let data = result;
      if (typeof result === 'string') {
        try {
          data = JSON.parse(result);
        } catch {
          data = { description: result };
        }
      }

      container.innerHTML = `
        <div class="weather-card weather-card--complete">
          <div class="weather-card__icon">
            ${getWeatherIcon(data?.condition || 'sunny')}
          </div>
          <div class="weather-card__content">
            <div class="weather-card__city">${city}</div>
            <div class="weather-card__temp">${data?.temperature || '--'}°</div>
            <div class="weather-card__desc">${data?.description || 'No data'}</div>
          </div>
        </div>
      `;
    },
    containerClass: 'agui-weather-widget',
  },
});

// Method 2: Register after the API is ready using the event
window.addEventListener('agui:tools:ready', (event) => {
  const { api } = event.detail;

  // Register a search results renderer
  api.register('search_documents', {
    render: ({ args, status, result, container }) => {
      if (status !== 'complete') {
        container.innerHTML = `
          <div class="search-results search-results--loading">
            <span class="search-results__icon">🔍</span>
            Searching for "${args.query || '...'}"
          </div>
        `;
        return;
      }

      // Parse results
      let results = [];
      if (typeof result === 'string') {
        try {
          results = JSON.parse(result);
        } catch {
          results = [];
        }
      } else if (Array.isArray(result)) {
        results = result;
      }

      if (results.length === 0) {
        container.innerHTML = `
          <div class="search-results search-results--empty">
            No results found for "${args.query}"
          </div>
        `;
        return;
      }

      container.innerHTML = `
        <div class="search-results">
          <div class="search-results__header">
            Found ${results.length} result${results.length !== 1 ? 's' : ''} for "${args.query}"
          </div>
          <ul class="search-results__list">
            ${results.map((r) => `
              <li class="search-results__item">
                <a href="${r.url || '#'}" class="search-results__link">
                  ${r.title || 'Untitled'}
                </a>
                ${r.snippet ? `<p class="search-results__snippet">${r.snippet}</p>` : ''}
              </li>
            `).join('')}
          </ul>
        </div>
      `;
    },
  });

  // Register a pattern-based renderer for all MCP Drupal tools
  api.register('/^mcp_drupal/', {
    render: ({ toolName, args, status, result, container }) => {
      const friendlyName = toolName
        .replace(/^mcp_drupal2?_/, '')
        .replace(/_/g, ' ')
        .replace(/\b\w/g, (c) => c.toUpperCase());

      if (status === 'pending' || status === 'streaming') {
        container.innerHTML = `
          <div class="drupal-action drupal-action--running">
            <span class="drupal-action__icon">⚙️</span>
            <span class="drupal-action__name">${friendlyName}</span>
            <span class="drupal-action__status">Running...</span>
          </div>
        `;
        return;
      }

      const isSuccess = status === 'complete';
      container.innerHTML = `
        <div class="drupal-action drupal-action--${isSuccess ? 'success' : 'error'}">
          <span class="drupal-action__icon">${isSuccess ? '✅' : '❌'}</span>
          <span class="drupal-action__name">${friendlyName}</span>
          <span class="drupal-action__status">${isSuccess ? 'Complete' : 'Failed'}</span>
        </div>
      `;
    },
    containerClass: 'agui-drupal-action',
  });
});

// Method 3: Direct registration if API is already available
if (window.AguiTools) {
  window.AguiTools.register('create_chart', {
    render: ({ args, status, result, container }) => {
      // Your chart rendering logic here
      container.innerHTML = `<div class="chart-placeholder">Chart: ${args.type || 'bar'}</div>`;
    },
  });
}

// Helper function for weather icons
function getWeatherIcon(condition) {
  const icons = {
    sunny: '☀️',
    cloudy: '☁️',
    rainy: '🌧️',
    stormy: '⛈️',
    snowy: '🌨️',
    windy: '💨',
    foggy: '🌫️',
  };
  return icons[condition?.toLowerCase()] || '🌤️';
}

// Add some basic styles for the custom components
const style = document.createElement('style');
style.textContent = `
  .weather-card {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px;
    min-width: 200px;
  }

  .weather-card--loading {
    background: linear-gradient(135deg, #89CFF0 0%, #B0E0E6 100%);
  }

  .weather-card--error {
    background: linear-gradient(135deg, #ff6b6b 0%, #ee5a5a 100%);
  }

  .weather-card__icon {
    font-size: 2.5rem;
  }

  .weather-card__city {
    font-weight: 600;
    font-size: 1.1rem;
  }

  .weather-card__temp {
    font-size: 1.5rem;
    font-weight: 700;
  }

  .weather-card__desc,
  .weather-card__loading,
  .weather-card__error {
    font-size: 0.9rem;
    opacity: 0.9;
  }

  .search-results {
    max-width: 100%;
  }

  .search-results--loading {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #666;
    font-style: italic;
  }

  .search-results__header {
    font-size: 0.875rem;
    color: #666;
    margin-bottom: 0.5rem;
  }

  .search-results__list {
    list-style: none;
    padding: 0;
    margin: 0;
  }

  .search-results__item {
    padding: 0.5rem 0;
    border-bottom: 1px solid #eee;
  }

  .search-results__item:last-child {
    border-bottom: none;
  }

  .search-results__link {
    color: #0066cc;
    text-decoration: none;
    font-weight: 500;
  }

  .search-results__link:hover {
    text-decoration: underline;
  }

  .search-results__snippet {
    margin: 0.25rem 0 0;
    font-size: 0.875rem;
    color: #666;
  }

  .drupal-action {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1rem;
    border-radius: 8px;
    font-size: 0.9rem;
  }

  .drupal-action--running {
    background: #e3f2fd;
    color: #1565c0;
  }

  .drupal-action--success {
    background: #e8f5e9;
    color: #2e7d32;
  }

  .drupal-action--error {
    background: #ffebee;
    color: #c62828;
  }

  .drupal-action__name {
    font-weight: 500;
  }

  .drupal-action__status {
    margin-left: auto;
    font-size: 0.8rem;
    opacity: 0.8;
  }
`;
document.head.appendChild(style);

