/**
 * DOM utilities for AG-UI Chat.
 *
 * Helper functions for creating and manipulating chat elements.
 */

import { type Message } from '@ag-ui/core';
import { renderMarkdown } from './markdown';

/**
 * Configuration for the chat component.
 */
export interface ChatConfig {
  endpoint: string;
  agentId: string;
  tokenEndpoint?: string;
  debug?: boolean;
}

/**
 * Parses chat configuration from a container element's data attributes.
 */
export function parseChatConfig(container: HTMLElement): ChatConfig {
  return {
    endpoint: container.dataset.endpoint || '/agui/api/chat',
    agentId: container.dataset.agentId || '',
    tokenEndpoint: container.dataset.tokenEndpoint || undefined,
    debug: container.dataset.debug === 'true',
  };
}

/**
 * Creates a message element for display in the chat.
 */
export function createMessageElement(message: Message): HTMLElement {
  const messageEl = document.createElement('div');
  messageEl.classList.add('agui-chat__message');
  messageEl.classList.add(`agui-chat__message--${message.role}`);
  messageEl.dataset.messageId = message.id;

  // Create content wrapper for styling
  const contentEl = document.createElement('div');
  contentEl.classList.add('agui-chat__message-content');

  // Get raw text and render as HTML (with basic markdown support)
  const text = getMessageText(message);
  contentEl.innerHTML = renderMarkdown(text);

  messageEl.appendChild(contentEl);
  return messageEl;
}

/**
 * Extracts text content from a Message object.
 * Handles both string content and array of content parts.
 */
export function getMessageText(message: Message): string {
  if (typeof message.content === 'string') {
    return message.content;
  }
  if (Array.isArray(message.content)) {
    return message.content
      .filter((item): item is { type: 'text'; text: string } => item.type === 'text')
      .map((item) => item.text)
      .join('');
  }
  return '';
}

/**
 * Updates a message element's content with new text.
 */
export function updateMessageContent(messageEl: HTMLElement, content: string): void {
  const contentEl = messageEl.querySelector('.agui-chat__message-content');
  if (contentEl) {
    contentEl.innerHTML = renderMarkdown(content);
  }
}

/**
 * Scrolls a container to the bottom.
 */
export function scrollToBottom(container: HTMLElement): void {
  container.scrollTop = container.scrollHeight;
}

/**
 * Required DOM elements for the chat component.
 */
export interface ChatElements {
  container: HTMLElement;
  messagesContainer: HTMLElement;
  form: HTMLFormElement;
  input: HTMLInputElement;
}

/**
 * Queries and validates required chat elements from a container.
 * Returns null if any required element is missing.
 */
export function queryChatElements(container: HTMLElement): ChatElements | null {
  const messagesContainer = container.querySelector('.agui-chat__messages');
  const form = container.querySelector('.agui-chat__input-form');
  const input = container.querySelector('.agui-chat__input');

  if (!messagesContainer || !form || !input) {
    console.error('AG-UI Chat: Missing required elements');
    return null;
  }

  return {
    container,
    messagesContainer: messagesContainer as HTMLElement,
    form: form as HTMLFormElement,
    input: input as HTMLInputElement,
  };
}

