/**
 * Status indicator utilities for AG-UI Chat.
 *
 * Provides animated status indicators for thinking, processing, and tool calls.
 */

export type IndicatorType = 'thinking' | 'processing' | 'tool-call';

export interface IndicatorConfig {
  type: IndicatorType;
  icon?: string;
  text: string;
}

/**
 * Default icons for each indicator type.
 */
const INDICATOR_ICONS: Record<IndicatorType, string | undefined> = {
  'thinking': undefined,
  'processing': '⚙️',
  'tool-call': '🔧',
};

/**
 * Creates an animated dots element for status indicators.
 */
function createAnimatedDots(): string {
  return `
    <span class="agui-chat__thinking-dots">
      <span>.</span><span>.</span><span>.</span>
    </span>
  `;
}

/**
 * Creates a status indicator element.
 */
export function createStatusIndicator(config: IndicatorConfig): HTMLElement {
  const indicator = document.createElement('div');
  indicator.classList.add('agui-chat__message', `agui-chat__message--${config.type}`);

  const icon = config.icon ?? INDICATOR_ICONS[config.type];

  const iconHtml = icon ? `<span class="agui-chat__${config.type}-icon">${icon}</span>` : '';
  const textClass = config.type === 'thinking' ? 'thinking-text' : `${config.type.replace('-', '-')}-text`;

  indicator.innerHTML = `
    ${iconHtml}
    <span class="agui-chat__${textClass}">${config.text}</span>
    ${createAnimatedDots()}
  `;

  return indicator;
}

/**
 * Manages a collection of status indicators.
 */
export class StatusIndicatorManager {
  private container: HTMLElement;
  private thinkingIndicator: HTMLElement | null = null;
  private processingIndicator: HTMLElement | null = null;
  private toolCallIndicators: Map<string, HTMLElement> = new Map();
  private scrollCallback: () => void;

  constructor(container: HTMLElement, scrollCallback: () => void) {
    this.container = container;
    this.scrollCallback = scrollCallback;
  }

  /**
   * Shows the thinking indicator.
   */
  showThinking(): void {
    if (this.thinkingIndicator) return;

    this.thinkingIndicator = createStatusIndicator({
      type: 'thinking',
      text: 'Thinking',
    });
    this.container.appendChild(this.thinkingIndicator);
    this.scrollCallback();
  }

  /**
   * Hides the thinking indicator.
   */
  hideThinking(): void {
    if (this.thinkingIndicator) {
      this.thinkingIndicator.remove();
      this.thinkingIndicator = null;
    }
  }

  /**
   * Shows the processing indicator.
   */
  showProcessing(): void {
    if (this.processingIndicator) return;

    this.processingIndicator = createStatusIndicator({
      type: 'processing',
      text: 'Processing results',
    });
    this.container.appendChild(this.processingIndicator);
    this.scrollCallback();
  }

  /**
   * Hides the processing indicator.
   */
  hideProcessing(): void {
    if (this.processingIndicator) {
      this.processingIndicator.remove();
      this.processingIndicator = null;
    }
  }

  /**
   * Shows a tool call indicator.
   */
  showToolCall(toolCallId: string, toolName: string): void {
    // Hide other indicators first
    this.hideThinking();
    this.hideProcessing();

    const indicator = createStatusIndicator({
      type: 'tool-call',
      text: `Calling <strong>${toolName}</strong> to help...`,
    });

    this.toolCallIndicators.set(toolCallId, indicator);
    this.container.appendChild(indicator);
    this.scrollCallback();
  }

  /**
   * Hides a specific tool call indicator.
   */
  hideToolCall(toolCallId: string): void {
    const indicator = this.toolCallIndicators.get(toolCallId);
    if (indicator) {
      indicator.remove();
      this.toolCallIndicators.delete(toolCallId);
    }
  }

  /**
   * Returns the number of active tool call indicators.
   */
  get activeToolCallCount(): number {
    return this.toolCallIndicators.size;
  }

  /**
   * Clears all indicators.
   */
  clearAll(): void {
    this.hideThinking();
    this.hideProcessing();
    this.toolCallIndicators.forEach((indicator) => indicator.remove());
    this.toolCallIndicators.clear();
  }
}

