/**
 * Token management utilities for AG-UI Chat.
 *
 * Handles fetching and caching JWT tokens for authenticated AG-UI requests.
 */

export interface TokenResponse {
  token: string;
  expires_at: number;
  expires_in: number;
}

export interface TokenConfig {
  /** URL to fetch token from (defaults to /agui/api/token) */
  tokenEndpoint?: string;
  /** Buffer time in seconds before expiry to refresh token (defaults to 60) */
  refreshBuffer?: number;
}

/**
 * Manages authentication tokens for AG-UI requests.
 */
export class TokenManager {
  private token: string | null = null;
  private expiresAt: number = 0;
  private tokenEndpoint: string;
  private refreshBuffer: number;
  private fetchPromise: Promise<string> | null = null;

  constructor(config: TokenConfig = {}) {
    this.tokenEndpoint = config.tokenEndpoint || '/agui/api/token';
    this.refreshBuffer = config.refreshBuffer || 60;
  }

  /**
   * Gets a valid token, fetching a new one if necessary.
   * Returns null if token fetching is disabled (no endpoint configured).
   */
  async getToken(): Promise<string | null> {
    // If we have a valid token, return it
    if (this.isTokenValid()) {
      return this.token;
    }

    // If a fetch is already in progress, wait for it
    if (this.fetchPromise) {
      return this.fetchPromise;
    }

    // Fetch a new token
    this.fetchPromise = this.fetchToken();

    try {
      const token = await this.fetchPromise;
      return token;
    } finally {
      this.fetchPromise = null;
    }
  }

  /**
   * Checks if the current token is still valid.
   */
  private isTokenValid(): boolean {
    if (!this.token) {
      return false;
    }

    const now = Math.floor(Date.now() / 1000);
    // Consider token invalid if it expires within the refresh buffer
    return this.expiresAt > now + this.refreshBuffer;
  }

  /**
   * Fetches a new token from the endpoint.
   */
  private async fetchToken(): Promise<string> {
    try {
      const response = await fetch(this.tokenEndpoint, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`Token fetch failed: ${response.status}`);
      }

      const data: TokenResponse = await response.json();

      if (data.error) {
        throw new Error(data.error as unknown as string);
      }

      this.token = data.token;
      this.expiresAt = data.expires_at;

      return this.token;
    } catch (error) {
      console.error('Failed to fetch AG-UI token:', error);
      throw error;
    }
  }

  /**
   * Clears the cached token.
   */
  clearToken(): void {
    this.token = null;
    this.expiresAt = 0;
  }

  /**
   * Checks if token authentication is enabled.
   * This can be used to conditionally skip token fetching for local endpoints.
   */
  isEnabled(): boolean {
    return Boolean(this.tokenEndpoint);
  }
}

/**
 * Creates authorization headers with the bearer token.
 */
export function createAuthHeaders(token: string | null): Record<string, string> {
  if (!token) {
    return {};
  }

  return {
    'Authorization': `Bearer ${token}`,
  };
}

