/**
 * Tool rendering utilities for AG-UI Chat.
 *
 * Provides a flexible system for rendering custom UI during tool calls.
 * External modules can register their own tool renderers via the global API.
 *
 * @example
 * // Register a custom tool renderer
 * window.AguiTools.register('get_weather', {
 *   render: ({ args, status, result, container }) => {
 *     container.innerHTML = `
 *       <div class="weather-card">
 *         <h3>Weather for ${args.city}</h3>
 *         ${status === 'complete' && result
 *           ? `<p>${result.temperature}°F - ${result.description}</p>`
 *           : '<p>Loading...</p>'
 *         }
 *       </div>
 *     `;
 *   },
 * });
 */

/**
 * Status of a tool call.
 */
export type ToolCallStatus = 'pending' | 'streaming' | 'complete' | 'error';

/**
 * Context provided to tool renderers.
 */
export interface ToolRenderContext<TArgs = Record<string, unknown>, TResult = unknown> {
  /** Unique ID for this tool call */
  toolCallId: string;
  /** Name of the tool being called */
  toolName: string;
  /** Tool arguments (may be partial during streaming) */
  args: TArgs;
  /** Current status of the tool call */
  status: ToolCallStatus;
  /** Result from the tool (only available when status is 'complete') */
  result?: TResult;
  /** Error message (only available when status is 'error') */
  error?: string;
  /** Container element to render into */
  container: HTMLElement;
  /** Callback to update the render (useful for animations) */
  requestUpdate: () => void;
}

/**
 * Configuration for a tool renderer.
 */
export interface ToolRendererConfig<TArgs = Record<string, unknown>, TResult = unknown> {
  /**
   * Render function called when the tool state changes.
   * Should update the container element with the appropriate UI.
   */
  render: (context: ToolRenderContext<TArgs, TResult>) => void;

  /**
   * Optional cleanup function called when the tool call is removed.
   */
  cleanup?: (context: ToolRenderContext<TArgs, TResult>) => void;

  /**
   * Whether to show the default tool indicator while rendering.
   * Defaults to false (custom render replaces default indicator).
   */
  showDefaultIndicator?: boolean;

  /**
   * CSS class to add to the container element.
   */
  containerClass?: string;
}

/**
 * Registration options for tool renderers.
 */
export interface ToolRegistrationOptions {
  /** Whether to override an existing renderer */
  override?: boolean;
}

/**
 * Active tool call state tracked by the registry.
 */
interface ActiveToolCall {
  toolCallId: string;
  toolName: string;
  args: Record<string, unknown>;
  status: ToolCallStatus;
  result?: unknown;
  error?: string;
  container: HTMLElement;
  config: ToolRendererConfig;
}

/**
 * Registry for tool renderers.
 * Manages registration and rendering of custom tool UIs.
 */
class ToolRendererRegistry {
  private renderers: Map<string, ToolRendererConfig> = new Map();
  private activeToolCalls: Map<string, ActiveToolCall> = new Map();
  private wildcardRenderers: Array<{
    pattern: RegExp;
    config: ToolRendererConfig;
  }> = [];

  /**
   * Registers a tool renderer.
   *
   * @param toolName - Name of the tool, or a regex pattern string (e.g., '/^mcp_/')
   * @param config - Renderer configuration
   * @param options - Registration options
   */
  register(
    toolName: string,
    config: ToolRendererConfig,
    options: ToolRegistrationOptions = {},
  ): void {
    // Check if this is a pattern (starts and ends with /)
    if (toolName.startsWith('/') && toolName.endsWith('/')) {
      const pattern = new RegExp(toolName.slice(1, -1));
      this.wildcardRenderers.push({ pattern, config });
      return;
    }

    const normalizedName = this.normalizeToolName(toolName);

    if (this.renderers.has(normalizedName) && !options.override) {
      console.warn(`AG-UI Tools: Renderer for "${toolName}" already exists. Use override option to replace.`);
      return;
    }

    this.renderers.set(normalizedName, config);
  }

  /**
   * Unregisters a tool renderer.
   */
  unregister(toolName: string): boolean {
    const normalizedName = this.normalizeToolName(toolName);
    return this.renderers.delete(normalizedName);
  }

  /**
   * Checks if a renderer is registered for a tool.
   */
  hasRenderer(toolName: string): boolean {
    return this.getRenderer(toolName) !== undefined;
  }

  /**
   * Gets the renderer for a tool, checking both exact matches and patterns.
   */
  getRenderer(toolName: string): ToolRendererConfig | undefined {
    const normalizedName = this.normalizeToolName(toolName);

    // Check exact match first
    const exactMatch = this.renderers.get(normalizedName);
    if (exactMatch) {
      return exactMatch;
    }

    // Check wildcard patterns
    for (const { pattern, config } of this.wildcardRenderers) {
      if (pattern.test(toolName)) {
        return config;
      }
    }

    return undefined;
  }

  /**
   * Creates a tool call container and starts rendering.
   * Returns the container element, or null if no renderer is registered.
   */
  startToolCall(
    toolCallId: string,
    toolName: string,
    args: Record<string, unknown> = {},
  ): HTMLElement | null {
    const config = this.getRenderer(toolName);
    if (!config) {
      return null;
    }

    // Create container element
    const container = document.createElement('div');
    container.classList.add('agui-chat__tool-render');
    container.dataset.toolCallId = toolCallId;
    container.dataset.toolName = toolName;

    if (config.containerClass) {
      container.classList.add(config.containerClass);
    }

    // Track the active tool call
    const activeCall: ActiveToolCall = {
      toolCallId,
      toolName,
      args,
      status: 'pending',
      container,
      config,
    };
    this.activeToolCalls.set(toolCallId, activeCall);

    // Initial render
    this.renderToolCall(activeCall);

    return container;
  }

  /**
   * Updates tool call arguments (during streaming).
   */
  updateToolCallArgs(toolCallId: string, args: Record<string, unknown>): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.args = args;
    activeCall.status = 'streaming';
    this.renderToolCall(activeCall);
  }

  /**
   * Completes a tool call with a result.
   */
  completeToolCall(toolCallId: string, result?: unknown): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.status = 'complete';
    activeCall.result = result;
    this.renderToolCall(activeCall);
  }

  /**
   * Marks a tool call as errored.
   */
  errorToolCall(toolCallId: string, error: string): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.status = 'error';
    activeCall.error = error;
    this.renderToolCall(activeCall);
  }

  /**
   * Removes a tool call and cleans up.
   */
  removeToolCall(toolCallId: string): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    // Call cleanup if defined
    if (activeCall.config.cleanup) {
      const context = this.createRenderContext(activeCall);
      activeCall.config.cleanup(context);
    }

    // Remove from DOM and tracking
    activeCall.container.remove();
    this.activeToolCalls.delete(toolCallId);
  }

  /**
   * Clears all active tool calls.
   */
  clearAll(): void {
    for (const toolCallId of this.activeToolCalls.keys()) {
      this.removeToolCall(toolCallId);
    }
  }

  /**
   * Gets all registered tool names.
   */
  getRegisteredTools(): string[] {
    return Array.from(this.renderers.keys());
  }

  /**
   * Normalizes a tool name for consistent lookup.
   */
  private normalizeToolName(toolName: string): string {
    return toolName.toLowerCase().trim();
  }

  /**
   * Renders a tool call using its registered renderer.
   */
  private renderToolCall(activeCall: ActiveToolCall): void {
    const context = this.createRenderContext(activeCall);
    try {
      activeCall.config.render(context);
    } catch (error) {
      console.error(`Error rendering tool "${activeCall.toolName}":`, error);
      // Show error state in the container
      activeCall.container.innerHTML = `
        <div class="agui-chat__tool-render--error" style="color: #c62828; padding: 0.5rem;">
          Error rendering tool: ${String(error)}
        </div>
      `;
    }
  }

  /**
   * Creates a render context for a tool call.
   */
  private createRenderContext(activeCall: ActiveToolCall): ToolRenderContext {
    return {
      toolCallId: activeCall.toolCallId,
      toolName: activeCall.toolName,
      args: activeCall.args,
      status: activeCall.status,
      result: activeCall.result,
      error: activeCall.error,
      container: activeCall.container,
      requestUpdate: () => this.renderToolCall(activeCall),
    };
  }
}

/**
 * Global tool renderer registry instance.
 */
export const toolRegistry = new ToolRendererRegistry();

/**
 * Public API for registering tool renderers.
 * Exposed on window.AguiTools for external use.
 */
export interface AguiToolsAPI {
  /**
   * Register a tool renderer.
   *
   * @example
   * AguiTools.register('get_weather', {
   *   render: ({ args, status, result, container }) => {
   *     // Render your UI into container
   *   }
   * });
   */
  register: typeof toolRegistry.register;

  /**
   * Unregister a tool renderer.
   */
  unregister: typeof toolRegistry.unregister;

  /**
   * Check if a renderer exists for a tool.
   */
  hasRenderer: typeof toolRegistry.hasRenderer;

  /**
   * Get all registered tool names.
   */
  getRegisteredTools: typeof toolRegistry.getRegisteredTools;
}

/**
 * Creates the public API object.
 */
export function createToolsAPI(): AguiToolsAPI {
  return {
    register: toolRegistry.register.bind(toolRegistry),
    unregister: toolRegistry.unregister.bind(toolRegistry),
    hasRenderer: toolRegistry.hasRenderer.bind(toolRegistry),
    getRegisteredTools: toolRegistry.getRegisteredTools.bind(toolRegistry),
  };
}

/**
 * Initializes the global AguiTools API.
 * Should be called once when the module loads.
 */
export function initializeToolsAPI(): void {
  if (typeof window !== 'undefined') {

    const api = createToolsAPI();
    (window as unknown as { AguiTools: AguiToolsAPI }).AguiTools = api;

    // Emit event so external code knows the API is ready
    window.dispatchEvent(new CustomEvent('agui:tools:ready', { detail: { api } }));

    // If there was a queue of pending registrations, process them
    const pendingQueue = (window as unknown as { AguiToolsQueue?: Array<{ name: string; config: ToolRendererConfig }> }).AguiToolsQueue;
    if (Array.isArray(pendingQueue)) {
      pendingQueue.forEach(({ name, config }) => {
        api.register(name, config);
      });
      delete (window as unknown as { AguiToolsQueue?: unknown }).AguiToolsQueue;
    }
  }
}

// Type declarations for global window object
declare global {
  interface Window {
    AguiTools?: AguiToolsAPI;
    AguiToolsQueue?: Array<{ name: string; config: ToolRendererConfig }>;
  }
}

