/**
 * @file
 * Weather tool renderer for AG-UI Chat demo.
 *
 * Renders a weather card UI when the get_weather tool is called.
 */

(function () {
  'use strict';

  /**
   * Weather condition icons mapping.
   */
  const WEATHER_ICONS = {
    sunny: '☀️',
    clear: '☀️',
    'partly cloudy': '⛅',
    cloudy: '☁️',
    overcast: '☁️',
    mist: '🌫️',
    fog: '🌫️',
    drizzle: '🌧️',
    rain: '🌧️',
    'light rain': '🌦️',
    'heavy rain': '🌧️',
    snow: '🌨️',
    'light snow': '🌨️',
    'heavy snow': '❄️',
    sleet: '🌨️',
    thunderstorm: '⛈️',
    storm: '⛈️',
    windy: '💨',
    default: '🌤️',
  };

  /**
   * Gets the appropriate weather icon for a condition.
   *
   * @param {string} condition - The weather condition string.
   * @returns {string} The emoji icon.
   */
  function getWeatherIcon(condition) {
    if (!condition) return WEATHER_ICONS.default;
    const normalized = condition.toLowerCase().trim();
    return WEATHER_ICONS[normalized] || WEATHER_ICONS.default;
  }

  /**
   * Formats temperature with unit.
   *
   * @param {number|string} temp - The temperature value.
   * @param {string} unit - The unit (C or F).
   * @returns {string} Formatted temperature.
   */
  function formatTemp(temp, unit = '°') {
    if (temp === undefined || temp === null) return '--';
    return `${Math.round(Number(temp))}${unit}`;
  }

  /**
   * Weather tool renderer configuration.
   */
  const weatherRenderer = {
    containerClass: 'agui-weather-widget',

    render: function ({ args, status, result, container }) {
      const location = args.location || args.city || 'Determining Location';

      // Loading state
      if (status === 'pending' || status === 'streaming') {
        container.innerHTML = `
          <div class="weather-card weather-card--loading">
            <div class="weather-card__icon">${WEATHER_ICONS.default}</div>
            <div class="weather-card__content">
              <div class="weather-card__location">${location}</div>
              <div class="weather-card__loading">
                <span class="weather-card__loading-text">Fetching weather</span>
                <span class="weather-card__loading-dots"><span>.</span><span>.</span><span>.</span></span>
              </div>
            </div>
          </div>
        `;
        return;
      }

      // Error state
      if (status === 'error') {
        container.innerHTML = `
          <div class="weather-card weather-card--error">
            <div class="weather-card__icon">⚠️</div>
            <div class="weather-card__content">
              <div class="weather-card__location">${location}</div>
              <div class="weather-card__error">Unable to fetch weather data</div>
            </div>
          </div>
        `;
        return;
      }

      // Parse result
      let data = result;
      if (typeof result === 'string') {
        try {
          data = JSON.parse(result);
        } catch (e) {
          data = { conditions: result };
        }
      }

      if (!data) {
        data = {};
      }

      // Use location from result if available (more accurate)
      const displayLocation = data.location || location;
      const conditions = data.conditions || 'Unknown';
      const icon = getWeatherIcon(conditions);

      // Build the complete weather card
      container.innerHTML = `
        <div class="weather-card weather-card--complete">
          <div class="weather-card__main">
            <div class="weather-card__icon">${icon}</div>
            <div class="weather-card__primary">
              <div class="weather-card__location">${displayLocation}</div>
              <div class="weather-card__temp">${formatTemp(data.temperature)}</div>
              <div class="weather-card__conditions">${conditions}</div>
            </div>
          </div>
          <div class="weather-card__details">
            <div class="weather-card__detail">
              <span class="weather-card__detail-label">Feels like</span>
              <span class="weather-card__detail-value">${formatTemp(data.feels_like)}</span>
            </div>
            <div class="weather-card__detail">
              <span class="weather-card__detail-label">Humidity</span>
              <span class="weather-card__detail-value">${data.humidity !== undefined ? `${data.humidity}%` : '--'}</span>
            </div>
            <div class="weather-card__detail">
              <span class="weather-card__detail-label">Wind</span>
              <span class="weather-card__detail-value">${data.wind_speed !== undefined ? `${data.wind_speed} km/h` : '--'}</span>
            </div>
            ${data.windGust ? `
            <div class="weather-card__detail">
              <span class="weather-card__detail-label">Gusts</span>
              <span class="weather-card__detail-value">${data.windGust} km/h</span>
            </div>
            ` : ''}
          </div>
        </div>
      `;
    },
  };

  // Register using the queue (works before API is ready)
  window.AguiToolsQueue = window.AguiToolsQueue || [];
  window.AguiToolsQueue.push({
    name: 'get_weather',
    config: weatherRenderer,
  });

  // Also register directly if API is already available
  if (window.AguiTools) {
    window.AguiTools.register('get_weather', weatherRenderer, { override: true });
  }

  // Register again when API becomes ready (handles race conditions)
  window.addEventListener('agui:tools:ready', function (event) {
    event.detail.api.register('get_weather', weatherRenderer, { override: true });
  });

})();

