<?php

namespace Drupal\agui\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Site\Settings;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Firebase\JWT\JWT;

/**
 * Controller for generating AG-UI authentication tokens.
 */
class AguiTokenController extends ControllerBase {

  /**
   * Default token expiry in seconds (15 minutes).
   */
  private const int DEFAULT_EXPIRY = 900;

  /**
   * Generates a short-lived JWT token for AG-UI requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the token.
   */
  public function generateToken(Request $request): JsonResponse {
    $secret = $this->getSecret();

    if (empty($secret)) {
      return new JsonResponse([
        'error' => 'AG-UI token secret not configured.',
      ], 500);
    }

    $now = time();
    $expiry = $now + $this->getExpiry();

    // Build JWT payload.
    $payload = [
      'iat' => $now,
      'exp' => $expiry,
      'sub' => $this->currentUser()->id(),
      'drupal_uid' => $this->currentUser()->id(),
      'drupal_roles' => $this->currentUser()->getRoles(),
    ];

    // Allow other modules to alter the payload.
    $this->moduleHandler()->alter('agui_token_payload', $payload, $request);

    // Encode and sign using firebase/php-jwt.
    $token = JWT::encode($payload, $secret, 'HS256');

    return new JsonResponse([
      'token' => $token,
      'expires_at' => $expiry,
      'expires_in' => $this->getExpiry(),
    ]);
  }

  /**
   * Gets the HMAC secret from settings.
   *
   * @return string|null
   *   The secret or NULL if not configured.
   */
  private function getSecret(): ?string {
    return Settings::get('agui_token_secret');
  }

  /**
   * Gets the token expiry time in seconds.
   *
   * @return int
   *   The expiry time in seconds.
   */
  private function getExpiry(): int {
    return Settings::get('agui_token_expiry', self::DEFAULT_EXPIRY);
  }

}
