<?php

namespace Drupal\agui\Controller;

use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\StreamedChatMessageIterator;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\ai_assistant_api\AiAssistantApiRunner;
use Drupal\ai_assistant_api\Data\UserMessage;
use Drupal\ai_assistant_api\Entity\AiAssistant;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Controller for AG-UI Chat.
 */
class AguiChatController extends ControllerBase {

  /**
   * The AI Assistant API runner.
   *
   * @var \Drupal\ai_assistant_api\AiAssistantApiRunner
   */
  protected $aiAssistantApiRunner;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * The function call plugin manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected $functionCallPluginManager;

  /**
   * Constructs a new AguiChatController object.
   *
   * @param \Drupal\ai_assistant_api\AiAssistantApiRunner $ai_assistant_api_runner
   *   The AI Assistant API runner.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   * @param \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager $function_call_plugin_manager
   *   The function call plugin manager.
   */
  public function __construct(
    AiAssistantApiRunner $ai_assistant_api_runner,
    EntityTypeManagerInterface $entity_type_manager,
    UuidInterface $uuid,
    FunctionCallPluginManager $function_call_plugin_manager,
  ) {
    $this->aiAssistantApiRunner = $ai_assistant_api_runner;
    $this->entityTypeManager = $entity_type_manager;
    $this->uuid = $uuid;
    $this->functionCallPluginManager = $function_call_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_assistant_api.runner'),
      $container->get('entity_type.manager'),
      $container->get('uuid'),
      $container->get('plugin.manager.ai.function_calls'),
    );
  }

  /**
   * Handle chat requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\StreamedResponse
   *   The response.
   */
  public function chat(Request $request): JsonResponse|StreamedResponse {
    $content = json_decode($request->getContent(), TRUE);

    // Try to get agentId from body, then query string.
    $agentId = $content['agentId'] ?? $request->query->get('agentId') ?? '';

    // Extract message from the messages array sent by HttpAgent.
    $messages = $content['messages'] ?? [];
    $messageText = '';

    // Find the last user message.
    // AG-UI messages have content as string or array of {type, text} parts.
    foreach (array_reverse($messages) as $msg) {
      if (isset($msg['role']) && $msg['role'] === 'user') {
        $msgContent = $msg['content'] ?? '';
        if (is_string($msgContent)) {
          $messageText = $msgContent;
        }
        elseif (is_array($msgContent)) {
          // Handle array of content parts [{type: "text", text: "..."}].
          foreach ($msgContent as $part) {
            if (is_array($part) && ($part['type'] ?? '') === 'text') {
              $messageText .= $part['text'] ?? '';
            }
            elseif (is_string($part)) {
              // Fallback for simple string arrays.
              $messageText .= $part;
            }
          }
        }
        break;
      }
    }

    // Fallback to direct 'message' property if provided (legacy/simple mode).
    if (empty($messageText)) {
      $messageText = $content['message'] ?? '';
    }

    if (empty($agentId)) {
      return new JsonResponse(['error' => 'Agent ID is required.'], 400);
    }

    $threadId = $content['threadId'] ?? '';
    if (!empty($threadId)) {
      $this->aiAssistantApiRunner->setThreadsKey($threadId);
    }

    $assistant = $this->entityTypeManager->getStorage('ai_assistant')->load($agentId);
    if (!$assistant instanceof AiAssistant) {
      // Fallback for demo purposes if the agent doesn't exist or for testing.
      if ($agentId === 'demo-agent') {
        return $this->streamDemoResponse($messageText);
      }
      return new JsonResponse(['error' => 'Invalid assistant ID.'], 404);
    }

    // Return a streaming response that processes the AI request.
    return $this->createAgentStreamResponse($assistant, $messageText);
  }

  /**
   * Creates a streaming response that processes the AI agent request.
   *
   * This method sends SSE events as the request is processed, allowing
   * the client to see "Thinking" status immediately.
   *
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $assistant
   *   The AI assistant entity.
   * @param string $messageText
   *   The user's message text.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The streamed response.
   */
  private function createAgentStreamResponse(AiAssistant $assistant, string $messageText): StreamedResponse {
    $response = new StreamedResponse();
    $response->headers->set('Content-Type', 'text/event-stream');
    $response->headers->set('Cache-Control', 'no-cache');
    $response->headers->set('Connection', 'keep-alive');
    $response->headers->set('X-Accel-Buffering', 'no');

    $response->setCallback(function () use ($assistant, $messageText) {
      // Disable ALL output buffering for true streaming.
      while (ob_get_level() > 0) {
        ob_end_flush();
      }
      if (function_exists('apache_setenv')) {
        apache_setenv('no-gzip', '1');
      }
      ini_set('zlib.output_compression', '0');
      ini_set('implicit_flush', '1');

      // Set up the AI assistant runner BEFORE sending any output.
      $this->aiAssistantApiRunner->setAssistant($assistant);
      $this->aiAssistantApiRunner->streamedOutput(TRUE);
      $this->aiAssistantApiRunner->setUserMessage(new UserMessage($messageText));
      $this->aiAssistantApiRunner->setThrowException(TRUE);

      // Start the session BEFORE any output (required for thread history).
      $this->aiAssistantApiRunner->startSession();

      $runId = $this->uuid->generate();
      $threadId = $this->uuid->generate();
      $messageId = $this->uuid->generate();

      // Send RUN_STARTED immediately so client sees "Thinking".
      $this->sendSseEvent([
        'type' => 'RUN_STARTED',
        'runId' => $runId,
        'threadId' => $threadId,
      ], TRUE);

      try {
        // Now process the AI request.
        $aiResponse = $this->aiAssistantApiRunner->process();
        $normalizedResponse = $aiResponse->getNormalized();

        $messageStarted = FALSE;
        $fullMessage = '';

        // Check if this is a ChatMessage with tool calls (agentic loop intermediate step).
        if ($normalizedResponse instanceof ChatMessage && !empty($normalizedResponse->getTools())) {
          // Emit tool call events for each tool.
          foreach ($normalizedResponse->getTools() as $tool) {
            $toolName = $tool->getName();
            $toolCallId = $this->uuid->generate();

            // Get friendly name from plugin if available.
            $friendlyName = $toolName;
            try {
              $plugin = $this->functionCallPluginManager->getFunctionCallFromFunctionName($toolName);
              if ($plugin) {
                $definition = $plugin->getPluginDefinition();
                $friendlyName = $definition['name'] ?? $toolName;
              }
            }
            catch (\Exception $e) {
              // Use the raw tool name if plugin lookup fails.
            }

            // Send TOOL_CALL_START event.
            $this->sendSseEvent([
              'type' => 'TOOL_CALL_START',
              'toolCallId' => $toolCallId,
              'toolCallName' => $friendlyName,
            ]);

            // Send TOOL_CALL_END event (tool already executed internally).
            $this->sendSseEvent([
              'type' => 'TOOL_CALL_END',
              'toolCallId' => $toolCallId,
            ]);
          }

          // Include any partial text from tool response.
          $partialText = $normalizedResponse->getText();
          if (!empty($partialText)) {
            $fullMessage = $partialText;
            $this->sendSseEvent([
              'type' => 'TEXT_MESSAGE_START',
              'messageId' => $messageId,
              'role' => 'assistant',
            ]);
            $messageStarted = TRUE;

            $chunks = str_split($partialText, 50);
            foreach ($chunks as $chunk) {
              $this->sendSseEvent([
                'type' => 'TEXT_MESSAGE_CONTENT',
                'messageId' => $messageId,
                'delta' => $chunk,
              ]);
              usleep(10000);
            }
          }
        }
        // Handle streaming response.
        elseif ($normalizedResponse instanceof StreamedChatMessageIterator) {
          foreach ($normalizedResponse as $chunk) {
            // Check for tool calls in streaming chunks.
            if (method_exists($chunk, 'getTools') && !empty($chunk->getTools())) {
              foreach ($chunk->getTools() as $tool) {
                $toolName = $tool->getName();
                $toolCallId = $this->uuid->generate();

                $friendlyName = $toolName;
                try {
                  $plugin = $this->functionCallPluginManager->getFunctionCallFromFunctionName($toolName);
                  if ($plugin) {
                    $definition = $plugin->getPluginDefinition();
                    $friendlyName = $definition['name'] ?? $toolName;
                  }
                }
                catch (\Exception $e) {
                  // Use raw name.
                }

                $this->sendSseEvent([
                  'type' => 'TOOL_CALL_START',
                  'toolCallId' => $toolCallId,
                  'toolCallName' => $friendlyName,
                ]);
                $this->sendSseEvent([
                  'type' => 'TOOL_CALL_END',
                  'toolCallId' => $toolCallId,
                ]);
              }
            }

            $text = $chunk->getText();
            if (!empty($text)) {
              $fullMessage .= $text;

              if (!$messageStarted) {
                $this->sendSseEvent([
                  'type' => 'TEXT_MESSAGE_START',
                  'messageId' => $messageId,
                  'role' => 'assistant',
                ]);
                $messageStarted = TRUE;
              }

              $this->sendSseEvent([
                'type' => 'TEXT_MESSAGE_CONTENT',
                'messageId' => $messageId,
                'delta' => $text,
              ]);
            }
          }
          // Ensure all streamed content is flushed before ending.
          if (function_exists('ob_flush') && ob_get_level() > 0) {
            @ob_flush();
          }
          flush();
          // Allow time for the flush to complete network transmission.
          usleep(50000);
        }
        // Handle non-streaming ChatMessage response (no tools).
        elseif ($normalizedResponse instanceof ChatMessage) {

          $text = $normalizedResponse->getText();
          if (!empty($text)) {
            $fullMessage = $text;

            $this->sendSseEvent([
              'type' => 'TEXT_MESSAGE_START',
              'messageId' => $messageId,
              'role' => 'assistant',
            ]);
            $messageStarted = TRUE;

            // Split into chunks for streaming effect.
            $chunks = str_split($text, 50);
            foreach ($chunks as $chunk) {
              $this->sendSseEvent([
                'type' => 'TEXT_MESSAGE_CONTENT',
                'messageId' => $messageId,
                'delta' => $chunk,
              ]);
              usleep(10000);
            }
          }
        }

        // Save assistant message for history.
        if (!empty($fullMessage)) {
          $this->aiAssistantApiRunner->setAssistantMessage($fullMessage);
        }

        // Send TEXT_MESSAGE_END if message was started.
        if ($messageStarted) {
          // Delay to ensure last content chunk is fully processed by client.
          usleep(100000);
          $this->sendSseEvent([
            'type' => 'TEXT_MESSAGE_END',
            'messageId' => $messageId,
          ]);
        }
      }
      catch (\Exception $e) {
        $this->getLogger('agui')->error('Error processing chat: @error', ['@error' => $e->getMessage()]);
        $this->sendSseEvent([
          'type' => 'RUN_ERROR',
          'message' => $e->getMessage(),
        ]);
      }

      // Send RUN_FINISHED event.
      $this->sendSseEvent([
        'type' => 'RUN_FINISHED',
        'runId' => $runId,
        'threadId' => $threadId,
      ]);
    });

    return $response;
  }


  /**
   * Stream a demo response.
   *
   * @param string $message
   *   The user message.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The streamed response.
   */
  private function streamDemoResponse(string $message): StreamedResponse {
    $response = new StreamedResponse();
    $response->headers->set('Content-Type', 'text/event-stream');
    $response->headers->set('Cache-Control', 'no-cache');
    $response->headers->set('Connection', 'keep-alive');
    $response->headers->set('X-Accel-Buffering', 'no');

    $response->setCallback(function () use ($message) {
      // Disable PHP output buffering.
      while (ob_get_level() > 0) {
        ob_end_flush();
      }
      ini_set('implicit_flush', '1');

      $runId = $this->uuid->generate();
      $threadId = $this->uuid->generate();
      $messageId = $this->uuid->generate();

      // Send RUN_STARTED event (with padding for proxy buffering).
      $this->sendSseEvent([
        'type' => 'RUN_STARTED',
        'runId' => $runId,
        'threadId' => $threadId,
      ], TRUE);

      $this->sendSseEvent([
        'type' => 'TEXT_MESSAGE_START',
        'messageId' => $messageId,
        'role' => 'assistant',
      ]);

      $demoText = "This is a demo response. You said: " . $message;
      $chunks = str_split($demoText, 5);

      foreach ($chunks as $chunk) {
        $this->sendSseEvent([
          'type' => 'TEXT_MESSAGE_CONTENT',
          'messageId' => $messageId,
          'delta' => $chunk,
        ]);
      }

      // Small delay to ensure last content chunk is processed by client.
      usleep(50000);
      $this->sendSseEvent([
        'type' => 'TEXT_MESSAGE_END',
        'messageId' => $messageId,
      ]);

      $this->sendSseEvent([
        'type' => 'RUN_FINISHED',
        'runId' => $runId,
        'threadId' => $threadId,
      ]);
    });

    return $response;
  }

  /**
   * Send SSE event.
   *
   * @param array $data
   *   The event data.
   * @param bool $isFirst
   *   Whether this is the first event (add padding for proxy buffering).
   */
  private function sendSseEvent(array $data, bool $isFirst = FALSE): void {
    // For the first event, send padding to overcome proxy buffering.
    // Nginx/FastCGI typically buffers 4KB-8KB before flushing.
    if ($isFirst) {
      // Send a comment with padding to force buffer flush.
      echo ": " . str_repeat(" ", 4096) . "\n\n";
      flush();
    }

    // Ensure we have valid data before encoding.
    if (empty($data) || !isset($data['type'])) {
      return;
    }

    // AG-UI protocol expects just the data line with JSON payload.
    // The 'type' field inside the JSON identifies the event type.
    $json = Json::encode($data);
    if ($json === FALSE || $json === 'null') {
      return;
    }

    echo "data: " . $json . "\n\n";

    // Ensure output is sent immediately.
    if (function_exists('ob_flush') && ob_get_level() > 0) {
      @ob_flush();
    }
    flush();
  }

}
