<?php

namespace Drupal\agui\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for selecting and rendering the AG-UI chat demo.
 */
class AguiDemoForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new AguiDemoForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'agui_demo_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check if we have values to render the chat.
    $endpoint = $form_state->getValue('endpoint');
    $agent_id = $form_state->getValue('agent_id');
    $source = $form_state->getValue('source') ?? 'drupal';

    // If Drupal source, get the selected assistant.
    if ($source === 'drupal') {
      $drupal_assistant = $form_state->getValue('drupal_assistant');
      if ($drupal_assistant) {
        $endpoint = '/agui/api/chat';
        $agent_id = $drupal_assistant;
      }
    }

    $form['#prefix'] = '<div id="agui-demo-wrapper">';
    $form['#suffix'] = '</div>';

    // Attach the weather tool renderer for demo purposes.
    $form['#attached']['library'][] = 'agui/weather_tool';

    $form['source'] = [
      '#type' => 'radios',
      '#title' => $this->t('Agent Source'),
      '#options' => [
        'drupal' => $this->t('Drupal AI Assistant'),
        'custom' => $this->t('Custom AG-UI Endpoint'),
      ],
      '#default_value' => $source,
      '#ajax' => [
        'callback' => '::ajaxRefresh',
        'wrapper' => 'agui-demo-wrapper',
      ],
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Debug Mode'),
      '#description' => $this->t('Enable debug mode for the chat component to log additional information to the console.'),
      '#default_value' => $form_state->getValue('debug') ?? FALSE,
    ];

    // Drupal AI Assistant selection.
    $form['drupal_settings'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="source"]' => ['value' => 'drupal'],
        ],
      ],
    ];

    $form['drupal_settings']['drupal_assistant'] = [
      '#type' => 'select',
      '#title' => $this->t('Select AI Assistant'),
      '#options' => $this->getAssistantOptions(),
      '#empty_option' => $this->t('- Select an assistant -'),
      '#default_value' => $form_state->getValue('drupal_assistant'),
      '#ajax' => [
        'callback' => '::ajaxLoadChat',
        'wrapper' => 'agui-chat-container',
      ],
    ];

    // Custom endpoint settings.
    $form['custom_settings'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="source"]' => ['value' => 'custom'],
        ],
      ],
    ];

    $form['custom_settings']['endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('AG-UI Endpoint URL'),
      '#description' => $this->t('The full URL to the AG-UI compatible endpoint (e.g., https://example.com/agui).'),
      '#default_value' => $form_state->getValue('endpoint'),
      '#placeholder' => 'https://example.com/agui',
    ];

    $form['custom_settings']['agent_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Agent ID'),
      '#description' => $this->t('The agent identifier for the AG-UI endpoint.'),
      '#default_value' => $form_state->getValue('agent_id'),
      '#placeholder' => 'my-agent',
    ];

    $form['custom_settings']['use_token'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use token authentication'),
      '#description' => $this->t('Enable HMAC-signed JWT authentication for secure communication with external endpoints.'),
      '#default_value' => $form_state->getValue('use_token') ?? TRUE,
    ];

    $form['custom_settings']['load_custom'] = [
      '#type' => 'button',
      '#value' => $this->t('Load Chat'),
      '#ajax' => [
        'callback' => '::ajaxLoadChat',
        'wrapper' => 'agui-chat-container',
      ],
      '#limit_validation_errors' => [],
    ];

    return $form;
  }

  /**
   * AJAX callback to refresh the form.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function ajaxRefresh(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * AJAX callback to load the chat component.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The chat container render array.
   */
  public function ajaxLoadChat(array &$form, FormStateInterface $form_state): array {
    $source = $form_state->getValue('source') ?? 'drupal';
    $endpoint = '';
    $agent_id = '';
    $token_endpoint = NULL;
    $debug = (bool) $form_state->getValue('debug');

    if ($source === 'drupal') {
      $drupal_assistant = $form_state->getValue('drupal_assistant');
      if ($drupal_assistant) {
        $endpoint = '/agui/api/chat';
        $agent_id = $drupal_assistant;
        // No token needed for Drupal internal endpoints.
      }
    }
    else {
      $endpoint = $form_state->getValue('endpoint');
      $agent_id = $form_state->getValue('agent_id');
      $use_token = $form_state->getValue('use_token');

      // Enable token auth for external endpoints if checkbox is checked.
      if ($use_token) {
        $token_endpoint = '/agui/api/token';
      }
    }

    $container = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'agui-chat-container',
        'class' => ['agui-demo-chat-container'],
      ],
    ];

    if (!empty($endpoint) && !empty($agent_id)) {
      $props = [
        'endpoint' => $endpoint,
        'agentId' => $agent_id,
        'debug' => $debug,
      ];

      if ($token_endpoint) {
        $props['tokenEndpoint'] = $token_endpoint;
      }

      // Add demo suggestions for the chat SDC.
      $props['suggestions'] = [
        ['title' => 'Features', 'message' => 'Tell me about your features and capabilities.'],
        ['title' => 'AG-UI', 'message' => 'Explain what AG-UI is.'],
        ['title' => 'Weather Tool', 'message' => 'What is the weather like in New York City today?'],
      ];

      $container['chat'] = [
        '#type' => 'component',
        '#component' => 'agui:chat',
        '#props' => $props,
      ];
    }
    else {
      $container['message'] = [
        '#markup' => '<p>' . $this->t('Please select an assistant or enter endpoint details.') . '</p>',
      ];
    }

    return $container;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Form is handled via AJAX, no submit needed.
  }

  /**
   * Gets available AI assistants as options.
   *
   * @return array
   *   An array of assistant options keyed by ID.
   */
  protected function getAssistantOptions(): array {
    $options = [];

    try {
      $storage = $this->entityTypeManager->getStorage('ai_assistant');
      $assistants = $storage->loadMultiple();

      foreach ($assistants as $assistant) {
        $options[$assistant->id()] = $assistant->label();
      }
    }
    catch (\Exception $e) {
      // AI Assistant API module may not be installed.
      $this->messenger()->addWarning($this->t('Could not load AI Assistants. Make sure the AI Assistant API module is enabled.'));
    }

    return $options;
  }

}

