/**
 * Human-in-the-loop utilities for AG-UI Chat.
 *
 * Handles interactions where the agent needs user confirmation or input
 * before proceeding. Agno uses `external_execution=True` on tools to
 * delegate execution to the UI for human review.
 *
 * Pattern:
 * 1. Agent emits TOOL_CALL_START/ARGS/END for an external_execution tool
 * 2. UI displays the tool args for user review (e.g., a list of steps)
 * 3. User approves/modifies/rejects
 * 4. UI sends TOOL_CALL_RESULT back to agent
 * 5. Agent continues with the result
 */

export type HitlActionType = 'confirm' | 'approve' | 'reject' | 'input' | 'steps';

/**
 * A step in a task plan (matches Agno's Step model).
 */
export interface TaskStep {
  description: string;
  status: 'enabled' | 'disabled' | 'executing';
}

/**
 * Represents a human-in-the-loop request from the agent.
 */
export interface HitlRequest {
  /** Unique ID for this HITL request (usually the toolCallId) */
  id: string;
  /** Type of action needed */
  type: HitlActionType;
  /** Tool name that triggered this HITL */
  toolName: string;
  /** Message to display to the user */
  message: string;
  /** Tool arguments (parsed) */
  args: Record<string, unknown>;
  /** Steps if this is a generate_task_steps call */
  steps?: TaskStep[];
}

/**
 * Represents the user's response to a HITL request.
 */
export interface HitlResponse {
  /** Tool call ID being responded to */
  toolCallId: string;
  /** Tool name */
  toolName: string;
  /** Whether the user approved */
  approved: boolean;
  /** The result to send back to the agent */
  result: string;
  /** Modified steps if applicable */
  steps?: TaskStep[];
}

/**
 * List of tool names that require human-in-the-loop interaction.
 * These are tools with external_execution=True in Agno.
 */
export const HITL_TOOL_NAMES = [
  'generate_task_steps',
  'confirm_action',
  'approve_plan',
  'human_approval',
  'request_confirmation',
];

/**
 * Checks if a tool call requires human-in-the-loop interaction.
 */
export function isHitlToolCall(toolName: string): boolean {
  return HITL_TOOL_NAMES.some(
    (name) => toolName.toLowerCase().includes(name.toLowerCase()),
  );
}

/**
 * Parses tool call arguments into a HITL request.
 */
export function createHitlRequestFromToolCall(
  toolCallId: string,
  toolName: string,
  args: Record<string, unknown>,
): HitlRequest {
  // Check if this is a steps-based tool
  const steps = args.steps as TaskStep[] | undefined;

  let message = 'Please review and approve this action.';
  let type: HitlActionType = 'approve';

  if (steps && Array.isArray(steps)) {
    type = 'steps';
    message = 'Review the following steps and toggle any you want to skip:';
  } else if (args.message) {
    message = String(args.message);
  }

  return {
    id: toolCallId,
    type,
    toolName,
    message,
    args,
    steps,
  };
}

/**
 * Creates the tool result string to send back to the agent.
 */
export function createHitlToolResult(response: HitlResponse): string {
  if (!response.approved) {
    return 'User rejected the request.';
  }

  if (response.steps) {
    const enabledSteps = response.steps.filter((s) => s.status === 'enabled');
    const disabledSteps = response.steps.filter((s) => s.status === 'disabled');
    return `User approved ${enabledSteps.length} steps. ${disabledSteps.length} steps were disabled by user.`;
  }

  return response.result || 'User approved the request.';
}

/**
 * Creates a HITL dialog element for step review.
 */
export function createHitlDialog(
  request: HitlRequest,
  onRespond: (response: HitlResponse) => void,
): HTMLElement {
  const dialog = document.createElement('div');
  dialog.classList.add('agui-chat__hitl-dialog');
  dialog.dataset.hitlId = request.id;

  const messageEl = document.createElement('div');
  messageEl.classList.add('agui-chat__hitl-message');
  messageEl.textContent = request.message;
  dialog.appendChild(messageEl);

  // If this is a steps review, render the steps
  if (request.type === 'steps' && request.steps) {
    const stepsContainer = document.createElement('div');
    stepsContainer.classList.add('agui-chat__hitl-steps');

    const stepElements: HTMLElement[] = [];

    request.steps.forEach((step, index) => {
      const stepEl = document.createElement('label');
      stepEl.classList.add('agui-chat__hitl-step');

      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.checked = step.status === 'enabled';
      checkbox.dataset.stepIndex = String(index);

      const stepText = document.createElement('span');
      stepText.classList.add('agui-chat__hitl-step-text');
      stepText.textContent = `${index + 1}. ${step.description}`;

      stepEl.appendChild(checkbox);
      stepEl.appendChild(stepText);
      stepsContainer.appendChild(stepEl);
      stepElements.push(stepEl);
    });

    dialog.appendChild(stepsContainer);
  }

  // Action buttons
  const actionsEl = document.createElement('div');
  actionsEl.classList.add('agui-chat__hitl-actions');

  const approveBtn = document.createElement('button');
  approveBtn.classList.add('agui-chat__hitl-btn', 'agui-chat__hitl-btn--approve');
  approveBtn.textContent = request.type === 'steps' ? 'Execute Selected Steps' : 'Approve';
  approveBtn.addEventListener('click', () => {
    // Gather the current state of steps if applicable
    let steps: TaskStep[] | undefined;
    if (request.type === 'steps' && request.steps) {
      const checkboxes = dialog.querySelectorAll<HTMLInputElement>('input[type="checkbox"]');
      steps = request.steps.map((step, index) => ({
        ...step,
        status: checkboxes[index]?.checked ? 'enabled' : 'disabled',
      })) as TaskStep[];
    }

    onRespond({
      toolCallId: request.id,
      toolName: request.toolName,
      approved: true,
      result: createHitlToolResult({ toolCallId: request.id, toolName: request.toolName, approved: true, result: '', steps }),
      steps,
    });
  });

  const rejectBtn = document.createElement('button');
  rejectBtn.classList.add('agui-chat__hitl-btn', 'agui-chat__hitl-btn--reject');
  rejectBtn.textContent = 'Cancel';
  rejectBtn.addEventListener('click', () => {
    onRespond({
      toolCallId: request.id,
      toolName: request.toolName,
      approved: false,
      result: 'User rejected the request.',
    });
  });

  actionsEl.appendChild(approveBtn);
  actionsEl.appendChild(rejectBtn);
  dialog.appendChild(actionsEl);

  return dialog;
}

// Legacy functions for backward compatibility with state-based HITL
export function parseHitlRequest(snapshot: unknown): HitlRequest | null {
  if (!snapshot || typeof snapshot !== 'object') {
    return null;
  }

  const state = snapshot as Record<string, unknown>;

  if (state.hitl_request || state.human_in_the_loop) {
    const hitl = (state.hitl_request || state.human_in_the_loop) as Record<string, unknown>;
    return {
      id: String(hitl.id || crypto.randomUUID()),
      type: (hitl.type as HitlActionType) || 'confirm',
      toolName: String(hitl.tool_name || 'unknown'),
      message: String(hitl.message || 'Please confirm to continue'),
      args: hitl.data as Record<string, unknown> || {},
    };
  }

  return null;
}

export function createHitlResponseState(response: HitlResponse): Record<string, unknown> {
  return {
    hitl_response: {
      tool_call_id: response.toolCallId,
      approved: response.approved,
      result: response.result,
    },
  };
}

