/**
 * Markdown and HTML rendering utilities for AG-UI Chat.
 *
 * Provides safe HTML rendering with support for:
 * - Allowed HTML tags: <strong>, <a>, <br>
 * - Markdown syntax: bold, italic, code, links
 * - External link icons
 */

/**
 * Escapes HTML special characters to prevent XSS.
 */
export function escapeHtml(text: string): string {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

/**
 * Renders text with markdown and allowed HTML tags.
 *
 * Allowed HTML tags are preserved and sanitized:
 * - <a href="...">text</a> - Links (opens in new tab with icon)
 * - <strong>text</strong> - Bold text
 * - <br> - Line breaks
 *
 * Markdown syntax supported:
 * - **bold** or __bold__
 * - *italic* or _italic_
 * - `inline code`
 * - ```code blocks```
 * - [link text](url)
 * - Newlines converted to <br>
 */
export function renderMarkdown(text: string): string {
  if (!text) return '';

  // Store allowed HTML elements with placeholders to preserve them through escaping
  const preserved: Map<string, string> = new Map();
  let placeholderIndex = 0;

  const createPlaceholder = (html: string): string => {
    const placeholder = `___PRESERVED_${placeholderIndex++}___`;
    preserved.set(placeholder, html);
    return placeholder;
  };

  let processed = text;

  // Fix malformed anchor tags (missing closing >)
  processed = processed.replace(
    /<a\s+([^>]*href="[^"]*"[^>]*)>([^<]*)<\/a(?![>])/gi,
    '<a $1>$2</a>',
  );

  // Preserve anchor tags - handle both with and without nested content
  processed = processed.replace(
    /<a\s+([^>]*href="([^"]*)"[^>]*)>([\s\S]*?)<\/a>/gi,
    (_match, _attrs, href, content) => {
      // Remove any existing ↗ icon from content to avoid duplication
      const cleanContent = content.replace(/\s*↗\s*/g, '').trim();
      // Strip any HTML tags from content for safety, but preserve text
      const textContent = cleanContent.replace(/<[^>]*>/g, '');
      const safeHtml = `<a href="${escapeHtml(href)}" target="_blank" rel="noopener">${escapeHtml(textContent)} <span class="agui-chat__link-icon">↗</span></a>`;
      return createPlaceholder(safeHtml);
    },
  );

  // Preserve strong tags with their content
  processed = processed.replace(
    /<strong>([\s\S]*?)<\/strong>/gi,
    (_match, content) => {
      // Check if content contains a placeholder (nested preserved element)
      if (content.includes('___PRESERVED_')) {
        return createPlaceholder(`<strong>${content}</strong>`);
      }
      return createPlaceholder(`<strong>${escapeHtml(content)}</strong>`);
    },
  );

  // Preserve br tags
  processed = processed.replace(/<br\s*\/?>/gi, () => createPlaceholder('<br>'));

  // Escape remaining HTML for safety
  let html = escapeHtml(processed);

  // Restore all preserved HTML
  preserved.forEach((original, placeholder) => {
    html = html.split(placeholder).join(original);
  });

  // Now apply markdown transformations

  // Code blocks (```code```)
  html = html.replace(
    /```(\w*)\n?([\s\S]*?)```/g,
    '<pre><code class="language-$1">$2</code></pre>',
  );

  // Inline code (`code`)
  html = html.replace(/`([^`]+)`/g, '<code>$1</code>');

  // Bold (**text** or __text__) - markdown style
  html = html.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
  html = html.replace(/__([^_]+)__/g, '<strong>$1</strong>');

  // Italic (*text* or _text_)
  html = html.replace(/\*([^*]+)\*/g, '<em>$1</em>');
  html = html.replace(/_([^_]+)_/g, '<em>$1</em>');

  // Links [text](url) - markdown style, add icon
  html = html.replace(
    /\[([^\]]+)\]\(([^)]+)\)/g,
    '<a href="$2" target="_blank" rel="noopener">$1 <span class="agui-chat__link-icon">↗</span></a>',
  );

  // Line breaks
  html = html.replace(/\n/g, '<br>');

  return html;
}

/**
 * Formats a tool/function name into a human-readable string.
 *
 * Converts snake_case or camelCase to Title Case with spaces.
 * Example: "search_articles" -> "Search articles"
 * Example: "getWeatherData" -> "Get weather data"
 */
export function formatToolName(toolName: string): string {
  return toolName
    .replace(/_/g, ' ')
    .replace(/([a-z])([A-Z])/g, '$1 $2')
    .toLowerCase()
    .replace(/^\w/, (c) => c.toUpperCase());
}

