/**
 * AG-UI Chat Component
 *
 * Implements AG-UI protocol for streaming chat with Drupal AI assistants.
 * Uses HttpAgent from @ag-ui/client for proper protocol compliance.
 */

import {
  HttpAgent,
  randomUUID,
  type AgentSubscriber,
} from '@ag-ui/client';
import { type Message } from '@ag-ui/core';
import {
  formatToolName,
  StatusIndicatorManager,
  type ChatConfig,
  type ChatElements,
  parseChatConfig,
  queryChatElements,
  createMessageElement,
  updateMessageContent,
  scrollToBottom,
  parseHitlRequest,
  createHitlResponseState,
  createHitlDialog,
  createHitlRequestFromToolCall,
  isHitlToolCall,
  type HitlResponse,
  type HitlRequest,
  TokenManager,
  createAuthHeaders,
  toolRegistry,
  initializeToolsAPI,
} from './utils';

/**
 * AG-UI Chat component class.
 */
class AguiChat {
  private config: ChatConfig;
  private elements: ChatElements;
  private agent: HttpAgent;
  private indicators: StatusIndicatorManager;
  private tokenManager: TokenManager | null = null;
  private isReady: boolean = false;

  // Current message state
  private currentAssistantMessage: HTMLElement | null = null;
  private currentMessageContent: string = '';
  private currentMessageId: string = '';
  private currentHitlDialog: HTMLElement | null = null;

  // HITL tool call tracking
  private pendingHitlToolCall: {
    toolCallId: string;
    toolName: string;
    args: Record<string, unknown>;
  } | null = null;

  // Custom tool render tracking
  private customRenderedTools: Set<string> = new Set();

  constructor(container: HTMLElement) {
    // Parse configuration from data attributes
    this.config = parseChatConfig(container);

    // Query required DOM elements
    const elements = queryChatElements(container);
    if (!elements) {
      throw new Error('AG-UI Chat: Missing required elements');
    }
    this.elements = elements;

    // Initialize status indicator manager
    this.indicators = new StatusIndicatorManager(
      this.elements.messagesContainer,
      () => this.scrollToBottom(),
    );

    // Initialize token manager if endpoint is configured
    if (this.config.tokenEndpoint) {
      this.tokenManager = new TokenManager({
        tokenEndpoint: this.config.tokenEndpoint,
      });
    }

    // Initialize the HttpAgent
    this.agent = this.createAgent();

    // Bind event handlers and initialize
    this.init();

    // --- SUGGESTION PILLS LOGIC ---
    const suggestionsContainer = container.querySelector('.agui-chat__suggestions');
    if (suggestionsContainer) {
      suggestionsContainer.addEventListener('click', (e) => {
        const target = e.target as HTMLElement;
        if (target.classList.contains('agui-chat__suggestion-pill')) {
          const message = target.getAttribute('data-message');
          if (message) {
            this.elements.input.value = message;
            // Optionally, trigger the send button click or form submit
            this.elements.form.dispatchEvent(new Event('submit', { bubbles: true, cancelable: true }));
          }
        }
      });
    }
    // --- END SUGGESTION PILLS LOGIC ---
  }

  /**
   * Creates and configures the HttpAgent.
   */
  private createAgent(): HttpAgent {
    const agentUrl = new URL(this.config.endpoint, window.location.origin);
    if (this.config.agentId) {
      agentUrl.searchParams.set('agentId', this.config.agentId);
    }

    return new HttpAgent({
      url: agentUrl.toString(),
      agentId: this.config.agentId,
      threadId: randomUUID(),
    });
  }

  /**
   * Initializes event listeners and prepares the chat.
   */
  private init(): void {
    this.elements.form.addEventListener('submit', (e) => {
      e.preventDefault();
      this.sendMessage();
    });

    // Initialize authentication if needed
    this.initializeAuth();
  }

  /**
   * Initializes authentication by pre-fetching a token.
   */
  private async initializeAuth(): Promise<void> {
    if (this.tokenManager) {
      try {
        this.setInputEnabled(false);
        this.updateSystemMessage('Authenticating...');
        await this.tokenManager.getToken();
        this.updateSystemMessage('Ready to chat.');
      } catch (error) {
        console.error('Failed to initialize auth:', error);
        this.updateSystemMessage('Authentication failed. Chat may not work with external endpoints.');
      } finally {
        this.setInputEnabled(true);
      }
    }
    this.isReady = true;
  }

  /**
   * Updates the system message in the chat.
   */
  private updateSystemMessage(message: string): void {
    const systemMessage = this.elements.messagesContainer.querySelector('.agui-chat__message--system');
    if (systemMessage) {
      systemMessage.textContent = message;
    }
  }

  /**
   * Sends a user message and runs the agent.
   */
  private async sendMessage(): Promise<void> {
    const text = this.elements.input.value.trim();
    if (!text) return;

    // Ensure we're ready
    if (!this.isReady) {
      return;
    }

    // Create and display user message
    const userMessage: Message = {
      id: randomUUID(),
      role: 'user',
      content: text,
    };

    this.agent.addMessage(userMessage);
    this.displayMessage(userMessage);
    this.elements.input.value = '';

    // Disable input while processing
    this.setInputEnabled(false);

    try {
      // Update agent headers with fresh token if using authentication
      if (this.tokenManager) {
        const token = await this.tokenManager.getToken();
        if (token) {
          // Create a new agent with updated headers
          this.updateAgentHeaders(createAuthHeaders(token));
        }
      }

      await this.runAgent();
    }
    catch (error) {
      console.error('Error sending message:', error);
      this.displaySystemError('Failed to get response');
    }
    finally {
      this.setInputEnabled(true);
      this.elements.input.focus();
    }
  }

  /**
   * Updates the agent's headers for authentication.
   */
  private updateAgentHeaders(headers: Record<string, string>): void {
    // HttpAgent stores headers in its config, we need to update them
    // The HttpAgent uses these headers in its requestInit method
    Object.assign(this.agent.headers, headers);
  }

  /**
   * Runs the agent with event subscription.
   */
  private async runAgent(): Promise<void> {
    this.resetMessageState();

    const subscriber = this.createSubscriber();
    await this.agent.runAgent({}, subscriber);
  }

  /**
   * Creates the event subscriber for the agent.
   */
  private createSubscriber(): AgentSubscriber {
    return {

      onEvent: ({event}) => {
        if (this.config.debug && event.type !== 'TEXT_MESSAGE_CONTENT') {
          console.log('Agent event:', event.type, event);
        }
      },

      onRunStartedEvent: () => {
        this.indicators.showThinking();
      },

      onTextMessageStartEvent: ({ event }) => {
        this.indicators.hideThinking();
        this.indicators.hideProcessing();
        this.resetMessageState();
        this.currentMessageId = event.messageId || randomUUID();
      },

      onTextMessageContentEvent: ({ textMessageBuffer }) => {
        this.handleTextContent(textMessageBuffer);
      },

      onTextMessageEndEvent: ({ textMessageBuffer }) => {
        this.finalizeMessage(textMessageBuffer);
      },

      onToolCallStartEvent: ({ event }) => {
        // Log this tool call.
        if (this.config.debug) {
          console.log('Tool call started:', event.toolCallName, event.toolCallId);
        }

        const friendlyName = formatToolName(event.toolCallName);

        // Check if this is a HITL tool (external_execution=True in Agno)
        if (isHitlToolCall(event.toolCallName)) {
          // Store the tool call info, we'll show the dialog when we get the args
          this.pendingHitlToolCall = {
            toolCallId: event.toolCallId,
            toolName: event.toolCallName,
            args: {},
          };
          this.indicators.hideThinking();
        }
        // Check if there's a custom renderer for this tool
        else if (toolRegistry.hasRenderer(event.toolCallName)) {
          const container = toolRegistry.startToolCall(
            event.toolCallId,
            event.toolCallName,
            {},
          );
          if (container) {
            this.customRenderedTools.add(event.toolCallId);
            this.elements.messagesContainer.appendChild(container);
            this.scrollToBottom();
            this.indicators.hideThinking();
          }
        } else {
          this.indicators.showToolCall(event.toolCallId, friendlyName);
        }
      },

      onToolCallArgsEvent: ({ event, partialToolCallArgs }) => {
        // Log tool call args event.
        if (this.config.debug) {
          console.log('Tool call args event:', event.toolCallId, partialToolCallArgs);
        }

        // If this is a HITL tool, update the args
        if (this.pendingHitlToolCall && this.pendingHitlToolCall.toolCallId === event.toolCallId) {
          this.pendingHitlToolCall.args = partialToolCallArgs;
        }
        // If this is a custom rendered tool, update the args for streaming render
        else if (this.customRenderedTools.has(event.toolCallId)) {
          toolRegistry.updateToolCallArgs(event.toolCallId, partialToolCallArgs);
          this.scrollToBottom();
        }
      },

      onToolCallEndEvent: ({ event, toolCallName, toolCallArgs }) => {
        // Log tool call end event.
        if (this.config.debug) {
          console.log('Tool call end event:', event.toolCallId, toolCallName, toolCallArgs);
        }

        // If this is a HITL tool, show the approval dialog
        if (this.pendingHitlToolCall && this.pendingHitlToolCall.toolCallId === event.toolCallId) {
          const hitlRequest = createHitlRequestFromToolCall(
            event.toolCallId,
            toolCallName,
            toolCallArgs,
          );
          this.showHitlDialog(hitlRequest);
          this.pendingHitlToolCall = null;
        }
        // If this is a custom rendered tool, finalize the args (result comes later)
        else if (this.customRenderedTools.has(event.toolCallId)) {
          toolRegistry.updateToolCallArgs(event.toolCallId, toolCallArgs);
          this.scrollToBottom();
        } else {
          this.indicators.hideToolCall(event.toolCallId);
          if (this.indicators.activeToolCallCount === 0) {
            this.indicators.showProcessing();
          }
        }
      },

      onToolCallResultEvent: ({ event }) => {
        // If this is a custom rendered tool, complete it with the result
        if (this.customRenderedTools.has(event.toolCallId)) {
          try {
            toolRegistry.completeToolCall(event.toolCallId, event.content);
          } catch (error) {
            if (this.config.debug) {
              console.error('Error completing tool render:', error);
            }
            toolRegistry.errorToolCall(event.toolCallId, String(error));
          }
          this.scrollToBottom();
        }
        // Keep processing indicator visible until text message starts
      },

      onRunFinishedEvent: () => {
        this.indicators.clearAll();
        // Clean up custom rendered tools tracking
        this.customRenderedTools.clear();
      },

      onRunErrorEvent: ({ event }) => {
        this.indicators.clearAll();
        this.removeHitlDialog();
        // Clean up custom rendered tools tracking
        this.customRenderedTools.clear();
        this.displaySystemError(event.message || 'An error occurred');
      },

      // Human-in-the-loop: Handle state snapshots for HITL requests
      onStateSnapshotEvent: ({ event }) => {
        this.handleStateSnapshot(event.snapshot);
      },

      onStateDeltaEvent: ({ event }) => {
        // State deltas could also contain HITL requests
        // For now, we primarily handle snapshots
        if (this.config.debug) {
          console.log('State delta received:', event.delta);
        }
      },
    };
  }

  /**
   * Handles incoming text content from the stream.
   */
  private handleTextContent(textMessageBuffer: string): void {
    // Create message element on first content
    if (!this.currentAssistantMessage && textMessageBuffer) {
      this.currentAssistantMessage = createMessageElement({
        id: this.currentMessageId,
        role: 'assistant',
        content: '',
      });
      this.elements.messagesContainer.appendChild(this.currentAssistantMessage);
    }

    // Update content
    if (this.currentAssistantMessage) {
      this.currentMessageContent = textMessageBuffer;
      updateMessageContent(this.currentAssistantMessage, this.currentMessageContent);
      this.scrollToBottom();
    }
  }

  /**
   * Finalizes the current message.
   */
  private finalizeMessage(textMessageBuffer: string): void {
    // Ensure final content is rendered before cleanup
    if (this.currentAssistantMessage && textMessageBuffer) {
      this.currentMessageContent = textMessageBuffer;
      updateMessageContent(this.currentAssistantMessage, this.currentMessageContent);
      this.scrollToBottom();
    }
    // Remove empty message bubbles
    else if (this.currentAssistantMessage && !textMessageBuffer) {
      this.currentAssistantMessage.remove();
    }
    this.resetMessageState();
  }

  /**
   * Resets the current message state.
   */
  private resetMessageState(): void {
    this.currentAssistantMessage = null;
    this.currentMessageContent = '';
    this.currentMessageId = '';
  }

  /**
   * Displays a system error message.
   */
  private displaySystemError(message: string): void {
    const errorMessage: Message = {
      id: randomUUID(),
      role: 'system',
      content: `Error: ${message}`,
    };
    this.displayMessage(errorMessage);
  }

  /**
   * Displays a message in the chat.
   */
  private displayMessage(message: Message): void {
    const messageEl = createMessageElement(message);
    this.elements.messagesContainer.appendChild(messageEl);
    this.scrollToBottom();
  }

  /**
   * Scrolls the messages container to the bottom.
   */
  private scrollToBottom(): void {
    scrollToBottom(this.elements.messagesContainer);
  }

  /**
   * Enables or disables the input field.
   */
  private setInputEnabled(enabled: boolean): void {
    this.elements.input.disabled = !enabled;
  }

  /**
   * Handles state snapshots for human-in-the-loop interactions (legacy pattern).
   */
  private handleStateSnapshot(snapshot: unknown): void {
    const hitlRequest = parseHitlRequest(snapshot);
    if (hitlRequest) {
      this.showHitlDialog(hitlRequest);
    }
  }

  /**
   * Shows a HITL dialog for user interaction.
   */
  private showHitlDialog(request: HitlRequest): void {
    // Hide any processing indicators
    this.indicators.clearAll();

    // Remove existing HITL dialog if any
    this.removeHitlDialog();

    // Create and display the HITL dialog
    this.currentHitlDialog = createHitlDialog(request, (response) => {
      this.handleHitlResponse(response, request);
    });

    this.elements.messagesContainer.appendChild(this.currentHitlDialog);
    this.scrollToBottom();
  }

  /**
   * Handles the user's response to a HITL request.
   */
  private handleHitlResponse(response: HitlResponse, request: HitlRequest): void {
    this.removeHitlDialog();

    // For tool-based HITL, we need to add a tool message with the result
    // This tells the agent the result of the external tool execution
    if (request.toolName) {
      const toolResultMessage: Message = {
        id: randomUUID(),
        role: 'tool',
        content: response.result,
        toolCallId: response.toolCallId,
      } as Message;

      // Add the tool result to the agent's messages
      this.agent.addMessage(toolResultMessage);

      // Show processing indicator while agent continues
      this.indicators.showProcessing();

      // Re-run the agent to continue with the tool result
      this.continueAgentAfterHitl();
    } else {
      // Legacy state-based HITL
      const responseState = createHitlResponseState(response);
      this.agent.setState(responseState);
      this.indicators.showProcessing();
    }
  }

  /**
   * Continues the agent after a HITL response.
   */
  private async continueAgentAfterHitl(): Promise<void> {
    try {
      const subscriber = this.createSubscriber();
      await this.agent.runAgent({}, subscriber);
    } catch (error) {
      console.error('Error continuing after HITL:', error);
      this.indicators.clearAll();
      this.displaySystemError('Failed to continue after approval');
    }
  }

  /**
   * Removes the current HITL dialog if present.
   */
  private removeHitlDialog(): void {
    if (this.currentHitlDialog) {
      this.currentHitlDialog.remove();
      this.currentHitlDialog = null;
    }
  }
}

/**
 * Initializes chat components within a context.
 * Tracks initialized elements to avoid double-initialization.
 */
function initializeChats(context: Document | HTMLElement = document): void {
  // Initialize the global tools API on first run
  initializeToolsAPI();

  const chats = context.querySelectorAll('.agui-chat:not([data-agui-initialized])');
  chats.forEach((chat) => {
    try {
      new AguiChat(chat as HTMLElement);
      chat.setAttribute('data-agui-initialized', 'true');
    }
    catch (error) {
      console.error('Failed to initialize AG-UI Chat:', error);
    }
  });
}

// Drupal behaviors for AJAX compatibility

if (typeof window.Drupal !== 'undefined' && window.Drupal.behaviors) {
  window.Drupal.behaviors.aguiChat = {
    attach: (context: Document | HTMLElement) => {
      initializeChats(context);
    },
  };
} else {
  // Fallback for non-Drupal environments
  document.addEventListener('DOMContentLoaded', () => initializeChats());
}

// Global type declarations
declare global {
  interface Window {
    Drupal?: {
      behaviors: Record<string, {
        attach?: (context: Document | HTMLElement, settings?: unknown) => void;
        detach?: (context: Document | HTMLElement, settings?: unknown, trigger?: string) => void;
      }>;
    };
  }
}

