# AG-UI

AG-UI provides an advanced graphical user interface for managing and interacting with AI models within the Drupal ecosystem.

## Components

### Chat (SDC: `agui:chat`)
An agentic chat component for integrating AI agents into Drupal templates. The chat SDC exposes props and a named slot so you can customize behaviour and markup. The component schema is authoritative — use the props below when rendering the SDC.

Usage (Twig example):
```twig
{# Basic usage with props #}
{% embed 'agui:chat' with {
  endpoint: '/agui/api/chat',
  agentId: 'default',
  tokenEndpoint: '/agui/api/token',
  suggestions: [
    { title: 'Show weather', message: 'What is the weather today?' },
    { title: 'Search site', message: 'Search for contributors' }
  ],
  debug: false
} %}
{% endembed %}

{# With tool renderers in the tools slot #}
{% embed 'agui:chat' with {
  endpoint: '/agui/api/chat',
  agentId: 'my_agent'
} %}
  {% block tools %}
    {# Embed tool SDCs that register custom UI renderers #}
    {% embed 'agui_examples:weather-tool' %}{% endembed %}
    {% embed 'agui_examples:research-tool' %}{% endembed %}
    {% embed 'my_module:custom-tool' %}{% endembed %}
  {% endblock %}
{% endembed %}
```

Props (component schema)
| Prop          | Type           | Default            | Description |
|---------------|----------------|--------------------|-------------|
| endpoint      | string         | `/agui/api/chat`   | API endpoint used by the chat component. Can be a Drupal route or an external URL when using a remote endpoint. |
| agentId       | string         | —                  | Identifier for the agent to use (matches an agent configured in the module). |
| tokenEndpoint | string         | `/agui/api/token`  | (Optional) token controller endpoint to request short-lived JWTs for authenticating direct/remote calls. |
| suggestions   | array<object>  | —                  | Optional array of suggestion objects rendered as pills. Each item: { title: string, message: string }.
| debug         | boolean        | false              | Enable debug mode for verbose logging.

Slots
- tools — Slot for tool renderer components. Add tool SDCs here to provide custom UIs for specific tools (e.g., weather cards, search results). Each tool SDC provides templates and JS that register with the chat.

## Tool Renderer Configuration

When registering a tool renderer via `window.AguiTools.register()`, you can configure how the tool's UI behaves:

### Renderer Properties

| Property             | Type     | Default | Description |
|----------------------|----------|---------|-------------|
| `render`             | Function | —       | (Required) Render function called when tool state changes. Receives a context object with tool data. |
| `cleanup`            | Function | —       | Optional cleanup function called when the tool call is removed. |
| `onStateEvent`       | Function | —       | Optional handler for state events (snapshots and deltas). Called whenever the agent sends state updates. |
| `targetContainerId`  | string   | —       | Optional ID of a DOM element where the tool should render. If specified and the element exists, the tool will render there instead of inline in the chat (enables "generative UI" patterns). |
| `clearTargetOnStart` | boolean  | `true`  | If true, clears the target container before rendering. Only applies when `targetContainerId` is used. |
| `containerClass`     | string   | —       | Optional CSS class to add to the tool's container element. |
| `showDefaultIndicator` | boolean | `false` | Whether to show the default tool indicator while rendering. Defaults to false (custom render replaces default indicator). |

### Example

```javascript
window.AguiTools.register('my_tool', {
  render: ({ args, status, result, container }) => {
    container.innerHTML = `<div class="my-tool">${result}</div>`;
  },
  containerClass: 'custom-tool-wrapper',
  showDefaultIndicator: true,
  targetContainerId: 'external-tools', // Render in external container instead of chat
  clearTargetOnStart: true,
  cleanup: ({ container }) => {
    // Clean up event listeners, etc.
  }
});
```

See the [examples README](examples/README.md) for detailed usage patterns and state management examples.

## API

- `POST /agui/api/chat`
  Endpoint used by the chat component to communicate with the backend. Payload and response shapes depend on your agent implementation; typically include message content, role, metadata and streaming chunks when enabled on the backend.

## Token controller (JWTs)

The module provides a token controller to issue short-lived JWTs for client-side use (for example when the chat frontend needs to call a remote endpoint directly). By default the token endpoint is `/agui/api/token` and it accepts authenticated requests from Drupal users to mint scoped tokens. Check the module's routing and permission definitions and the examples README for usage patterns and configuration.

## Demo behaviour

The bundle includes a demo that shows two modes:
- Drupal assistant — the chat UI posts to the local Drupal agent endpoint and the server-side agent runs inside Drupal; useful for local testing and when you want Drupal to mediate access to contrib integrations.
- Remote endpoint — the chat UI calls an external/remote endpoint directly (for example a hosted assistant or proxy). When using remote endpoints, the token controller can provide short-lived JWTs so the browser can authenticate safely.

## Tools and examples

This README intentionally keeps tooling notes light — for examples of tool integrations (tools, tool selectors, or plugin usage) see the module examples README at `modules/custom/agui/examples/README.md`.

## Notes

- Use the SDC name `agui:chat` in templates.
- Ensure backend routes and permissions are configured for the `/agui/api/chat` and token endpoints when used.
- Prefer using the demo and examples to validate agent configuration before wiring a production endpoint.
