/**
 * AG-UI State Management
 *
 * Implements state synchronization between agents and frontends using:
 * - STATE_SNAPSHOT events for complete state updates
 * - STATE_DELTA events for incremental updates (JSON Patch format)
 *
 * Provides a shared state architecture where:
 * - Agents can access the application's current state
 * - Frontends can observe and react to agent state changes
 * - Both sides can modify state for collaborative workflows
 *
 * @see https://docs.ag-ui.com/concepts/state
 */

import { applyPatch, type Operation } from 'fast-json-patch';

/**
 * Represents a JSON Patch operation (RFC 6902).
 */
export type JsonPatchOperation = Operation;

/**
 * State snapshot event data.
 */
export interface StateSnapshotEventData {
  type: 'STATE_SNAPSHOT';
  snapshot: unknown;
}

/**
 * State delta event data.
 */
export interface StateDeltaEventData {
  type: 'STATE_DELTA';
  delta: JsonPatchOperation[];
}

/**
 * Combined state event type.
 */
export type StateEventData = StateSnapshotEventData | StateDeltaEventData;

/**
 * Callback for state change listeners.
 */
export type StateChangeListener = (
  state: unknown,
  event: StateEventData,
  previousState: unknown,
) => void;

/**
 * Callback for UI state update requests.
 * Used when the UI wants to send state changes to the agent.
 */
export type StateUpdateCallback = (delta: JsonPatchOperation[]) => void;

/**
 * Configuration for the state manager.
 */
export interface StateManagerConfig {
  /** Initial state */
  initialState?: unknown;
  /** Enable debug logging */
  debug?: boolean;
  /** Callback when UI wants to update agent state */
  onStateUpdate?: StateUpdateCallback;
}

/**
 * Manages AG-UI shared state between frontend and agent.
 *
 * Supports:
 * - Receiving state snapshots and deltas from the agent
 * - Notifying registered listeners of state changes
 * - Sending state updates from UI back to the agent
 */
export class StateManager {
  private state: unknown = {};
  private listeners: Set<StateChangeListener> = new Set();
  private debug: boolean;
  private onStateUpdate?: StateUpdateCallback;

  constructor(config: StateManagerConfig = {}) {
    this.state = config.initialState ?? {};
    this.debug = config.debug ?? false;
    this.onStateUpdate = config.onStateUpdate;
  }

  /**
   * Gets the current state.
   */
  getState<T = unknown>(): T {
    return this.state as T;
  }

  /**
   * Handles a STATE_SNAPSHOT event from the agent.
   * Replaces the entire state with the snapshot.
   */
  handleSnapshot(snapshot: unknown): void {
    const previousState = this.state;
    this.state = snapshot;

    const eventData: StateSnapshotEventData = {
      type: 'STATE_SNAPSHOT',
      snapshot,
    };

    this.notifyListeners(eventData, previousState);
  }

  /**
   * Handles a STATE_DELTA event from the agent.
   * Applies JSON Patch operations to the current state.
   */
  handleDelta(delta: JsonPatchOperation[]): void {
    const previousState = this.state;

    try {
      // Apply JSON Patch operations
      const result = applyPatch(
        this.state,
        delta,
        /* validate */ true,
        /* mutate */ false,
      );
      this.state = result.newDocument;

      const eventData: StateDeltaEventData = {
        type: 'STATE_DELTA',
        delta,
      };

      if (this.debug) {
        console.log('AG-UI State: Delta applied', delta, 'New state:', this.state);
      }

      this.notifyListeners(eventData, previousState);
    } catch (error) {
      console.warn(
        'AG-UI State: Failed to apply state patch:\n' +
        `Current state: ${JSON.stringify(this.state, null, 2)}\n` +
        `Patch operations: ${JSON.stringify(delta, null, 2)}\n` +
        `Error: ${error}`,
      );
    }
  }

  /**
   * Handles any state event (snapshot or delta).
   */
  handleEvent(event: StateEventData): void {
    if (event.type === 'STATE_SNAPSHOT') {
      this.handleSnapshot(event.snapshot);
    } else if (event.type === 'STATE_DELTA') {
      this.handleDelta(event.delta);
    }
  }

  /**
   * Subscribes to state changes.
   * Returns an unsubscribe function.
   */
  subscribe(listener: StateChangeListener): () => void {
    this.listeners.add(listener);
    return () => this.listeners.delete(listener);
  }

  /**
   * Sends a state update from the UI to the agent.
   * This allows the frontend to modify the shared state.
   *
   * @param delta - JSON Patch operations to apply
   */
  sendUpdate(delta: JsonPatchOperation[]): void {
    if (this.onStateUpdate) {
      if (this.debug) {
        console.log('AG-UI State: Sending update to agent', delta);
      }
      this.onStateUpdate(delta);
    } else {
      console.warn('AG-UI State: No onStateUpdate callback configured');
    }
  }

  /**
   * Convenience method to set a value in the state and send to agent.
   *
   * @param path - JSON Pointer path (e.g., '/user/preference')
   * @param value - Value to set
   */
  setValue(path: string, value: unknown): void {
    const delta: JsonPatchOperation[] = [
      { op: 'replace', path, value },
    ];

    // Apply locally first
    this.handleDelta(delta);

    // Send to agent
    this.sendUpdate(delta);
  }

  /**
   * Convenience method to add a value to the state and send to agent.
   *
   * @param path - JSON Pointer path
   * @param value - Value to add
   */
  addValue(path: string, value: unknown): void {
    const delta: JsonPatchOperation[] = [
      { op: 'add', path, value },
    ];

    this.handleDelta(delta);
    this.sendUpdate(delta);
  }

  /**
   * Convenience method to remove a value from the state and send to agent.
   *
   * @param path - JSON Pointer path
   */
  removeValue(path: string): void {
    const delta: JsonPatchOperation[] = [
      { op: 'remove', path },
    ];

    this.handleDelta(delta);
    this.sendUpdate(delta);
  }

  /**
   * Resets the state to an empty object.
   */
  reset(): void {
    this.state = {};
    this.listeners.clear();
  }

  /**
   * Notifies all listeners of a state change.
   */
  private notifyListeners(event: StateEventData, previousState: unknown): void {
    for (const listener of this.listeners) {
      try {
        listener(this.state, event, previousState);
      } catch (error) {
        console.error('AG-UI State: Error in state listener', error);
      }
    }
  }
}

