import { createStatusIndicator } from './indicators';
import type { StateEventData, JsonPatchOperation } from './state';

/**
 * Tool rendering utilities for AG-UI Chat.
 *
 * Provides a flexible system for rendering custom UI during tool calls.
 * External modules can register their own tool renderers via the global API.
 *
 * @example
 * // Register a tool renderer that displays inline in the chat
 * window.AguiTools.register('get_weather', {
 *   render: ({ args, status, result, container }) => {
 *     container.innerHTML = `
 *       <div class="weather-card">
 *         <h3>Weather for ${args.city}</h3>
 *         ${status === 'complete' && result
 *           ? `<p>${result.temperature}°F - ${result.description}</p>`
 *           : '<p>Loading...</p>'
 *         }
 *       </div>
 *     `;
 *   },
 * });
 *
 * @example
 * // Register a tool that renders in an external container (generative UI)
 * window.AguiTools.register('show_product', {
 *   targetContainerId: 'product-preview-panel',
 *   render: ({ args, status, result, container }) => {
 *     container.innerHTML = `<product-card sku="${args.sku}" loading="${status !== 'complete'}"></product-card>`;
 *   },
 * });
 */

/**
 * Status of a tool call.
 */
export type ToolCallStatus = 'pending' | 'streaming' | 'complete' | 'error';

/**
 * Context provided to tool renderers.
 */
export interface ToolRenderContext<TArgs = Record<string, unknown>, TResult = unknown> {
  /** Unique ID for this tool call */
  toolCallId: string;
  /** Name of the tool being called */
  toolName: string;
  /** Tool arguments (may be partial during streaming) */
  args: TArgs;
  /** Current status of the tool call */
  status: ToolCallStatus;
  /** Result from the tool (only available when status is 'complete') */
  result?: TResult;
  /** Error message (only available when status is 'error') */
  error?: string;
  /** Container element to render into */
  container: HTMLElement;
  /** Callback to update the render (useful for animations) */
  requestUpdate: () => void;
  /** Whether rendering in an external container (true) or inline in chat (false) */
  isExternal: boolean;
  /** Current shared state from the agent */
  state: unknown;
  /**
   * Send a state update to the agent.
   * @param delta - JSON Patch operations to apply
   */
  sendStateUpdate: (delta: JsonPatchOperation[]) => void;
  /**
   * Send a message to the agent (triggers a new agent run).
   * Use this for actions like "Dig Deeper" that need to prompt the agent.
   * @param message - The message text to send
   */
  sendMessage: (message: string) => void;
}

/**
 * Configuration for a tool renderer.
 */
export interface ToolRendererConfig<TArgs = Record<string, unknown>, TResult = unknown> {
  /**
   * Render function called when the tool state changes.
   * Should update the container element with the appropriate UI.
   */
  render: (context: ToolRenderContext<TArgs, TResult>) => void;

  /**
   * Optional cleanup function called when the tool call is removed.
   */
  cleanup?: (context: ToolRenderContext<TArgs, TResult>) => void;

  /**
   * Optional handler for state events (snapshots and deltas).
   * Called whenever the agent sends state updates.
   * Use this to react to shared state changes.
   *
   * @example
   * AguiTools.register('collaborative_editor', {
   *   render: ({ container, state }) => {
   *     container.innerHTML = `<div>Document: ${state?.document?.title}</div>`;
   *   },
   *   onStateEvent: ({ state, event, context }) => {
   *     if (event.type === 'STATE_SNAPSHOT') {
   *       console.log('Full state received:', state);
   *     } else {
   *       console.log('State patches:', event.delta);
   *     }
   *   }
   * });
   */
  onStateEvent?: (params: {
    state: unknown;
    event: StateEventData;
    previousState: unknown;
    context: ToolRenderContext<TArgs, TResult>;
  }) => void;

  /**
   * Whether to show the default tool indicator while rendering.
   * Defaults to false (custom render replaces default indicator).
   */
  showDefaultIndicator?: boolean;

  /**
   * CSS class to add to the container element.
   */
  containerClass?: string;

  /**
   * Optional ID of a DOM element where the tool should render.
   * If specified and the element exists, the tool will render there instead of inline in the chat.
   * This enables "generative UI" patterns where tools can update specific areas of the page.
   *
   * @example
   * // Tool renders into a sidebar panel
   * AguiTools.register('show_product', {
   *   targetContainerId: 'product-preview-panel',
   *   render: ({ args, container }) => {
   *     container.innerHTML = `<product-card sku="${args.sku}"></product-card>`;
   *   }
   * });
   */
  targetContainerId?: string;

  /**
   * If true, clears the target container before rendering.
   * Only applies when targetContainerId is used.
   * Defaults to true.
   */
  clearTargetOnStart?: boolean;
}

/**
 * Registration options for tool renderers.
 */
export interface ToolRegistrationOptions {
  /** Whether to override an existing renderer */
  override?: boolean;
}

/**
 * Active tool call state tracked by the registry.
 */
interface ActiveToolCall {
  toolCallId: string;
  toolName: string;
  args: Record<string, unknown>;
  status: ToolCallStatus;
  result?: unknown;
  error?: string;
  container: HTMLElement;
  config: ToolRendererConfig;
  /** Whether this tool renders in an external container (not inline in chat) */
  isExternal: boolean;
}

/**
 * Result from starting a tool call.
 */
export interface StartToolCallResult {
  /** The container element the tool renders into */
  container: HTMLElement;
  /**
   * Whether the tool renders externally (in a targetContainerId).
   * If true, the chat should NOT append this container to messages.
   */
  isExternal: boolean;
}

/**
 * Registry for tool renderers.
 * Manages registration and rendering of custom tool UIs.
 */
class ToolRendererRegistry {
  private renderers: Map<string, ToolRendererConfig> = new Map();
  private activeToolCalls: Map<string, ActiveToolCall> = new Map();
  private wildcardRenderers: Array<{
    pattern: RegExp;
    config: ToolRendererConfig;
  }> = [];

  /** Current shared state */
  private currentState: unknown = {};

  /** Callback to send state updates to the agent */
  private sendStateUpdateCallback?: (delta: JsonPatchOperation[]) => void;

  /** Callback to send a message to the agent */
  private sendMessageCallback?: (message: string) => void;

  /**
   * Sets the callback for sending state updates to the agent.
   * Called by the chat component when initializing.
   */
  setStateUpdateCallback(callback: (delta: JsonPatchOperation[]) => void): void {
    this.sendStateUpdateCallback = callback;
  }

  /**
   * Sets the callback for sending messages to the agent.
   * Called by the chat component when initializing.
   */
  setSendMessageCallback(callback: (message: string) => void): void {
    this.sendMessageCallback = callback;
  }

  /**
   * Handles a state event from the agent.
   * Notifies all active tool renderers that have onStateEvent handlers.
   */
  handleStateEvent(state: unknown, event: StateEventData, previousState: unknown): void {
    this.currentState = state;

    // Notify all active tool calls with onStateEvent handler
    for (const activeCall of this.activeToolCalls.values()) {
      if (activeCall.config.onStateEvent) {
        try {
          const context = this.createRenderContext(activeCall);
          activeCall.config.onStateEvent({
            state,
            event,
            previousState,
            context,
          });
        } catch (error) {
          console.error(`AG-UI Tools: Error in state handler for tool "${activeCall.toolName}"`, error);
        }
      }
    }
  }

  /**
   * Gets the current shared state (internal use).
   */
  getState<T = unknown>(): T {
    return this.currentState as T;
  }

  /**
   * Sends a state update to the agent (internal use).
   */
  sendStateUpdate(delta: JsonPatchOperation[]): void {
    if (this.sendStateUpdateCallback) {
      this.sendStateUpdateCallback(delta);
    } else {
      console.warn('AG-UI Tools: No state update callback configured');
    }
  }

  /**
   * Sends a message to the agent, triggering a new agent run (internal use).
   */
  sendMessage(message: string): void {
    if (this.sendMessageCallback) {
      this.sendMessageCallback(message);
    } else {
      console.warn('AG-UI Tools: No send message callback configured');
    }
  }

  /**
   * Registers a tool renderer.
   *
   * @param toolName - Name of the tool, or a regex pattern string (e.g., '/^mcp_/')
   * @param config - Renderer configuration
   * @param options - Registration options
   */
  register(
    toolName: string,
    config: ToolRendererConfig,
    options: ToolRegistrationOptions = {},
  ): void {
    // Check if this is a pattern (starts and ends with /)
    if (toolName.startsWith('/') && toolName.endsWith('/')) {
      const pattern = new RegExp(toolName.slice(1, -1));
      this.wildcardRenderers.push({ pattern, config });
      return;
    }

    const normalizedName = this.normalizeToolName(toolName);

    if (this.renderers.has(normalizedName) && !options.override) {
      console.warn(`AG-UI Tools: Renderer for "${toolName}" already exists. Use override option to replace.`);
      return;
    }

    this.renderers.set(normalizedName, config);
  }

  /**
   * Unregisters a tool renderer.
   */
  unregister(toolName: string): boolean {
    const normalizedName = this.normalizeToolName(toolName);
    return this.renderers.delete(normalizedName);
  }

  /**
   * Checks if a renderer is registered for a tool.
   */
  hasRenderer(toolName: string): boolean {
    return this.getRenderer(toolName) !== undefined;
  }

  /**
   * Checks if a tool is configured to render in an external container.
   */
  isExternalRenderer(toolName: string): boolean {
    const config = this.getRenderer(toolName);
    if (!config?.targetContainerId) {
      return false;
    }
    // Also verify the target element exists
    return document.getElementById(config.targetContainerId) !== null;
  }

  /**
   * Gets the renderer for a tool, checking both exact matches and patterns.
   */
  getRenderer(toolName: string): ToolRendererConfig | undefined {
    const normalizedName = this.normalizeToolName(toolName);

    // Check exact match first
    const exactMatch = this.renderers.get(normalizedName);
    if (exactMatch) {
      return exactMatch;
    }

    // Check wildcard patterns
    for (const { pattern, config } of this.wildcardRenderers) {
      if (pattern.test(toolName)) {
        return config;
      }
    }

    return undefined;
  }

  /**
   * Creates a tool call container and starts rendering.
   * Returns the result with container and render location info, or null if no renderer is registered.
   */
  startToolCall(
    toolCallId: string,
    toolName: string,
    args: Record<string, unknown> = {},
  ): StartToolCallResult | null {
    const config = this.getRenderer(toolName);
    if (!config) {
      return null;
    }

    let container: HTMLElement;
    let isExternal = false;

    // Check if tool wants to render in an external container
    if (config.targetContainerId) {
      const targetElement = document.getElementById(config.targetContainerId);
      if (targetElement) {
        isExternal = true;

        // Clear target if configured (default: true)
        if (config.clearTargetOnStart !== false) {
          targetElement.innerHTML = '';
        }

        // Create a wrapper inside the target for this specific tool call
        container = document.createElement('div');
        container.classList.add('agui-chat__tool-render');
        container.dataset.toolCallId = toolCallId;
        container.dataset.toolName = toolName;

        if (config.containerClass) {
          container.classList.add(config.containerClass);
        }

        targetElement.appendChild(container);
      } else {
        // Target not found, fall back to inline rendering
        console.warn(`AG-UI Tools: Target container "${config.targetContainerId}" not found for tool "${toolName}". Falling back to inline rendering.`);
        container = this.createInlineContainer(toolCallId, toolName, config);
      }
    } else {
      // Standard inline rendering in chat
      container = this.createInlineContainer(toolCallId, toolName, config);
    }

    // Track the active tool call
    const activeCall: ActiveToolCall = {
      toolCallId,
      toolName,
      args,
      status: 'pending',
      container,
      config,
      isExternal,
    };
    this.activeToolCalls.set(toolCallId, activeCall);

    // Initial render
    this.renderToolCall(activeCall);

    return { container, isExternal };
  }

  /**
   * Creates an inline container element for rendering in the chat.
   */
  private createInlineContainer(
    toolCallId: string,
    toolName: string,
    config: ToolRendererConfig,
  ): HTMLElement {
    const container = document.createElement('div');
    container.classList.add('agui-chat__tool-render');
    container.dataset.toolCallId = toolCallId;
    container.dataset.toolName = toolName;

    if (config.containerClass) {
      container.classList.add(config.containerClass);
    }

    return container;
  }

  /**
   * Updates tool call arguments (during streaming).
   */
  updateToolCallArgs(toolCallId: string, args: Record<string, unknown>): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.args = args;
    activeCall.status = 'streaming';
    this.renderToolCall(activeCall);
  }

  /**
   * Completes a tool call with a result.
   */
  completeToolCall(toolCallId: string, result?: unknown): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.status = 'complete';
    activeCall.result = result;
    this.renderToolCall(activeCall);
  }

  /**
   * Marks a tool call as errored.
   */
  errorToolCall(toolCallId: string, error: string): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    activeCall.status = 'error';
    activeCall.error = error;
    this.renderToolCall(activeCall);
  }

  /**
   * Removes a tool call and cleans up.
   */
  removeToolCall(toolCallId: string): void {
    const activeCall = this.activeToolCalls.get(toolCallId);
    if (!activeCall) return;

    // Call cleanup if defined
    if (activeCall.config.cleanup) {
      const context = this.createRenderContext(activeCall);
      activeCall.config.cleanup(context);
    }

    // Remove from DOM and tracking
    activeCall.container.remove();
    this.activeToolCalls.delete(toolCallId);
  }

  /**
   * Clears all active tool calls.
   */
  clearAll(): void {
    for (const toolCallId of this.activeToolCalls.keys()) {
      this.removeToolCall(toolCallId);
    }
  }

  /**
   * Gets all registered tool names.
   */
  getRegisteredTools(): string[] {
    return Array.from(this.renderers.keys());
  }

  /**
   * Normalizes a tool name for consistent lookup.
   */
  private normalizeToolName(toolName: string): string {
    return toolName.toLowerCase().trim();
  }

  /**
   * Renders a tool call using its registered renderer.
   */
  private renderToolCall(activeCall: ActiveToolCall): void {
    const context = this.createRenderContext(activeCall);
    try {
      activeCall.config.render(context);
    } catch (error) {
      console.error(`Error rendering tool "${activeCall.toolName}":`, error);
      // Show error state in the container
      activeCall.container.innerHTML = `
        <div class="agui-chat__tool-render--error" style="color: #c62828; padding: 0.5rem;">
          Error rendering tool: ${String(error)}
        </div>
      `;
    }
  }

  /**
   * Creates a render context for a tool call.
   */
  private createRenderContext(activeCall: ActiveToolCall): ToolRenderContext {
    return {
      toolCallId: activeCall.toolCallId,
      toolName: activeCall.toolName,
      args: activeCall.args,
      status: activeCall.status,
      result: activeCall.result,
      error: activeCall.error,
      container: activeCall.container,
      requestUpdate: () => this.renderToolCall(activeCall),
      isExternal: activeCall.isExternal,
      state: this.currentState,
      sendStateUpdate: (delta: JsonPatchOperation[]) => this.sendStateUpdate(delta),
      sendMessage: (message: string) => this.sendMessage(message),
    };
  }
}

/**
 * Global tool renderer registry instance.
 */
export const toolRegistry = new ToolRendererRegistry();

/**
 * Public API for registering tool renderers.
 * Exposed on window.AguiTools for external use.
 */
export interface AguiToolsAPI {
  /**
   * Register a tool renderer.
   *
   * Tools can interact with AG-UI state through:
   * - `context.state` - current state in render function
   * - `context.sendStateUpdate(delta)` - send updates to agent
   * - `onStateEvent` callback - react to state changes
   *
   * @example
   * // Tool with rendering
   * AguiTools.register('get_weather', {
   *   render: ({ args, status, result, container, state }) => {
   *     container.innerHTML = `<div>Weather: ${result}</div>`;
   *   }
   * });
   *
   * @example
   * // State-only tool (no visual rendering, just reacts to state)
   * AguiTools.register('state_tracker', {
   *   render: () => {}, // No-op render
   *   onStateEvent: ({ state, event }) => {
   *     console.log('State changed:', state);
   *   }
   * });
   */
  register: typeof toolRegistry.register;

  /**
   * Unregister a tool renderer.
   */
  unregister: typeof toolRegistry.unregister;

  /**
   * Check if a renderer exists for a tool.
   */
  hasRenderer: typeof toolRegistry.hasRenderer;

  /**
   * Check if a tool is configured to render in an external container.
   * Returns true if the tool has a targetContainerId and that element exists in the DOM.
   */
  isExternalRenderer: typeof toolRegistry.isExternalRenderer;

  /**
   * Get all registered tool names.
   */
  getRegisteredTools: typeof toolRegistry.getRegisteredTools;

  /**
   * Indicator utilities for tool renderers.
   * @example
   * const el = AguiTools.indicators.createStatusIndicator({ type: 'tool-call', text: 'Searching...' });
   * // See IndicatorConfig type in indicators.ts for config options
   */
  indicators: {
    createStatusIndicator: typeof createStatusIndicator;
  };
}

/**
 * Creates the public API object.
 */
export function createToolsAPI(): AguiToolsAPI {
  return {
    register: toolRegistry.register.bind(toolRegistry),
    unregister: toolRegistry.unregister.bind(toolRegistry),
    hasRenderer: toolRegistry.hasRenderer.bind(toolRegistry),
    isExternalRenderer: toolRegistry.isExternalRenderer.bind(toolRegistry),
    getRegisteredTools: toolRegistry.getRegisteredTools.bind(toolRegistry),
    indicators: {
      createStatusIndicator,
    },
  };
}

let toolsAPIInitialized = false;

/**
 * Initializes the global AguiTools API.
 * Should be called once when the module loads.
 */
export function initializeToolsAPI(): void {
  if (toolsAPIInitialized) {
    return;
  }
  toolsAPIInitialized = true;
  if (typeof window !== 'undefined') {
    const api = createToolsAPI();
    (window as unknown as { AguiTools: AguiToolsAPI }).AguiTools = api;

    // Emit event so external code knows the API is ready
    window.dispatchEvent(new CustomEvent('agui:tools:ready', { detail: { api } }));

    // If there was a queue of pending registrations, process them
    const pendingQueue = (window as unknown as { AguiToolsQueue?: Array<{ name: string; config: ToolRendererConfig; options: ToolRegistrationOptions }> }).AguiToolsQueue;
    if (Array.isArray(pendingQueue)) {
      pendingQueue.forEach(({ name, config, options }) => {
        api.register(name, config, options);
      });
      delete (window as unknown as { AguiToolsQueue?: unknown }).AguiToolsQueue;
    }
  }
}

// Type declarations for global window object
declare global {
  interface Window {
    AguiTools?: AguiToolsAPI;
    AguiToolsQueue?: Array<{ name: string; config: ToolRendererConfig; options?: ToolRegistrationOptions }>;
  }
}

