/**
 * @file
 * Weather tool renderer for AG-UI Chat.
 *
 * Uses <template> elements from the Twig file to render weather cards.
 * This keeps HTML in Twig and makes the tool themable/customizable.
 */

(function () {
  'use strict';

  /**
   * Weather condition icons mapping.
   */
  const WEATHER_ICONS = {
    sunny: '☀️',
    clear: '☀️',
    'partly cloudy': '⛅',
    cloudy: '☁️',
    overcast: '☁️',
    mist: '🌫️',
    fog: '🌫️',
    drizzle: '🌧️',
    rain: '🌧️',
    'light rain': '🌦️',
    'heavy rain': '🌧️',
    snow: '🌨️',
    'light snow': '🌨️',
    'heavy snow': '❄️',
    sleet: '🌨️',
    thunderstorm: '⛈️',
    storm: '⛈️',
    windy: '💨',
    default: '🌤️',
  };

  /**
   * Gets the weather icon for a condition.
   *
   * @param {string} condition - The weather condition string.
   * @returns {string} The emoji icon.
   */
  function getWeatherIcon(condition) {
    if (!condition) return WEATHER_ICONS.default;
    const normalized = condition.toLowerCase().trim();
    return WEATHER_ICONS[normalized] || WEATHER_ICONS.default;
  }

  /**
   * Formats temperature with degree symbol.
   *
   * @param {number|string} temp - The temperature value.
   * @returns {string} Formatted temperature.
   */
  function formatTemp(temp) {
    if (temp === undefined || temp === null) return '--';
    return `${Math.round(Number(temp))}°`;
  }

  /**
   * Gets a template from the component's templates.
   *
   * @param {string} templateName - The template name (loading, error,
   *   complete).
   * @returns {DocumentFragment|null} The cloned template content.
   */
  function getTemplate(templateName) {
    const templateEl = document.querySelector(
      `template[data-weather-template="${templateName}"]`
    );
    if (!templateEl) {
      console.warn(`Weather tool template "${templateName}" not found.`);
      return null;
    }
    return templateEl.content.cloneNode(true);
  }

  /**
   * Weather tool renderer configuration.
   */
  const weatherRenderer = {
    containerClass: 'agui-weather-widget',

    render({ args, status, result, container }) {
      const location = args.location || args.city || 'Determining Location';

      // Loading state.
      if (status === 'pending' || status === 'streaming') {
        const template = getTemplate('loading');
        if (template) {
          const locationEl = template.querySelector('[data-weather-location]');
          if (locationEl) locationEl.textContent = location;
          container.innerHTML = '';
          container.appendChild(template);
        }
        return;
      }

      // Error state.
      if (status === 'error') {
        const template = getTemplate('error');
        if (template) {
          const locationEl = template.querySelector('[data-weather-location]');
          if (locationEl) locationEl.textContent = location;
          container.innerHTML = '';
          container.appendChild(template);
        }
        return;
      }

      // Parse result.
      let data = result;
      if (typeof result === 'string') {
        try {
          data = JSON.parse(result);
        } catch {
          data = { conditions: result };
        }
      }
      data = data || {};

      // Complete state.
      const template = getTemplate('complete');
      if (!template) return;

      // Populate template with data.
      const displayLocation = data.location || location;
      const conditions = data.conditions || 'Unknown';
      const icon = getWeatherIcon(conditions);

      const setEl = (selector, value) => {
        const el = template.querySelector(selector);
        if (el) el.textContent = value;
      };

      setEl('[data-weather-icon]', icon);
      setEl('[data-weather-location]', displayLocation);
      setEl('[data-weather-temp]', formatTemp(data.temperature));
      setEl('[data-weather-conditions]', conditions);
      setEl('[data-weather-feels-like]', formatTemp(data.feels_like));
      setEl('[data-weather-humidity]', data.humidity !== undefined ? `${data.humidity}%` : '--');
      setEl('[data-weather-wind]', data.wind_speed !== undefined ? `${data.wind_speed} km/h` : '--');

      // Handle optional gusts field.
      const gustsContainer = template.querySelector('[data-weather-gusts-container]');
      if (gustsContainer && data.windGust) {
        gustsContainer.removeAttribute('hidden');
        setEl('[data-weather-gusts]', `${data.windGust} km/h`);
      }

      container.innerHTML = '';
      container.appendChild(template);
    },
  };

  // Register the weather tool renderer when AG-UI tools are ready.
  window.addEventListener('agui:tools:ready', (event) => {
    event.detail.api.register('get_weather', weatherRenderer, {override: true});
  });

})();

