# Tool Renderer Examples

This folder contains examples showing different ways to create custom UI renderers
for AG-UI Chat tools.

## Two Approaches

### 1. Simple: JavaScript-Only Tool Renderer

**File:** `simple-tool-renderer.js`

Best for quick, lightweight tools where you want everything in one file.
Uses inline HTML in JavaScript.

**Pros:**
- Single file, easy to share
- No Twig/CSS dependencies
- Works as a standalone library

**Cons:**
- HTML is mixed with JavaScript
- Harder to theme/customize
- Styles must be injected dynamically

### 2. Complex: SDC (Single Directory Component) Tool Renderer

**Example:** `components/weather-tool/`

Best for production tools that need:
- Clean separation of concerns (HTML, CSS, JS)
- Drupal theming/overriding
- Proper asset management

**Pros:**
- HTML in Twig templates (themable)
- CSS as a separate file (overridable)
- Follows Drupal SDC patterns
- Uses `<template>` elements for clean JS

**Cons:**
- More files to manage
- Requires SDC knowledge

## Usage

### Simple Tool (Library Attachment)

```php
$build['#attached']['library'][] = 'my_module/my_tool_renderer';
```

### Complex Tool (SDC in Chat Slot)

In Twig templates:

```twig
{% embed 'agui:chat' with {
  endpoint: '/agui/api/chat',
  agentId: 'my_agent'
} %}
  {% block tools %}
    {# Embed tool SDCs that register custom UI renderers #}
    {% embed 'agui_examples:weather-tool' %}{% endembed %}
    {% embed 'my_module:my-tool' %}{% endembed %}
  {% endblock %}
{% endembed %}
```

In PHP (render array):

```php
$build['chat'] = [
  '#type' => 'component',
  '#component' => 'agui:chat',
  '#props' => [...],
  '#slots' => [
    'tools' => [
      ['#type' => 'component', '#component' => 'my_module:my-tool'],
    ],
  ],
];
```

## Registration Methods

All tools register using the same AG-UI Tools API:

```javascript
// Method 1: Queue (before API is ready)
window.AguiToolsQueue = window.AguiToolsQueue || [];
window.AguiToolsQueue.push({
  name: 'tool_name',
  config: { render: (ctx) => { ... } },
});

// Method 2: Direct (if API is available)
if (window.AguiTools) {
  window.AguiTools.register('tool_name', { render: (ctx) => { ... } });
}

// Method 3: Event listener (guaranteed to work)
window.addEventListener('agui:tools:ready', (e) => {
  e.detail.api.register('tool_name', { render: (ctx) => { ... } });
});
```

## Render Context

The `render` function receives a context object:

```javascript
{
  toolCallId: string,      // Unique ID for this tool call
  toolName: string,        // The tool being called
  args: object,            // Arguments passed to the tool
  status: string,          // 'pending' | 'streaming' | 'complete' | 'error'
  result: any,             // Tool result (when complete)
  error: string,           // Error message (when status is 'error')
  container: HTMLElement,  // DOM element to render into
  requestUpdate: Function, // Call to re-render the tool
  isExternal: boolean,     // Whether rendering in an external container
  state: any,              // Current shared state from the agent
  sendStateUpdate: Function, // Send state updates to the agent
  sendMessage: Function    // Send a message to trigger a new agent run
}
```

## Renderer Configuration

When registering a tool, you can configure various aspects of how it renders:

```javascript
window.AguiTools.register('my_tool', {
  // Required: render function
  render: ({ args, status, result, container }) => {
    container.innerHTML = `<div>${result}</div>`;
  },

  // Optional: cleanup function called when tool is removed
  cleanup: ({ container }) => {
    // Remove event listeners, timers, etc.
  },

  // Optional: react to state changes from the agent
  onStateEvent: ({ state, event, context }) => {
    if (event.type === 'STATE_DELTA') {
      console.log('State changed:', event.delta);
    }
    context.requestUpdate(); // Re-render with new state
  },

  // Optional: render in an external container instead of inline in chat
  targetContainerId: 'external-tools',

  // Optional: clear the target container before rendering (default: true)
  clearTargetOnStart: true,

  // Optional: add a CSS class to the container
  containerClass: 'my-custom-tool',

  // Optional: show default indicator alongside custom render (default: false)
  showDefaultIndicator: false,
});
```

### Configuration Properties

| Property             | Type     | Default | Description |
|----------------------|----------|---------|-------------|
| `render`             | Function | —       | (Required) Render function called when tool state changes. |
| `cleanup`            | Function | —       | Optional cleanup function called when the tool call is removed. |
| `onStateEvent`       | Function | —       | Optional handler for state events (snapshots and deltas). |
| `targetContainerId`  | string   | —       | Optional ID of a DOM element where the tool should render externally. |
| `clearTargetOnStart` | boolean  | `true`  | If true, clears the target container before rendering. |
| `containerClass`     | string   | —       | Optional CSS class to add to the tool's container element. |
| `showDefaultIndicator` | boolean | `false` | Whether to show the default tool indicator while rendering. |

## State Management

Tools can interact with AG-UI shared state, enabling real-time synchronization
between agents and frontends. See [State.md](../State.md) for full details.

### Accessing State in Render

The current agent state is available via `context.state`:

```javascript
AguiTools.register('my_tool', {
  render: ({ container, state }) => {
    // Access current state
    const userPref = state?.user?.preference;
    container.innerHTML = `<div>Preference: ${userPref}</div>`;
  }
});
```

### Sending State Updates

Tools can send state updates back to the agent using JSON Patch operations (RFC 6902):

```javascript
AguiTools.register('approval_tool', {
  render: ({ container, args, sendStateUpdate }) => {
    container.innerHTML = `
      <div class="approval-dialog">
        <p>Approve action: ${args.action}?</p>
        <button class="approve">Approve</button>
        <button class="reject">Reject</button>
      </div>
    `;

    container.querySelector('.approve')?.addEventListener('click', () => {
      sendStateUpdate([
        { op: 'replace', path: '/approved', value: true },
        { op: 'add', path: '/approvedAt', value: Date.now() }
      ]);
    });

    container.querySelector('.reject')?.addEventListener('click', () => {
      sendStateUpdate([
        { op: 'replace', path: '/approved', value: false }
      ]);
    });
  }
});
```

### Reacting to State Changes

Use the `onStateEvent` callback to react when the agent updates state:

```javascript
AguiTools.register('progress_tracker', {
  render: ({ container, state }) => {
    const progress = state?.task?.progress || 0;
    container.innerHTML = `
      <div class="progress-bar">
        <div class="fill" style="width: ${progress}%"></div>
      </div>
    `;
  },

  onStateEvent: ({ state, event, context }) => {
    // Called when agent sends STATE_SNAPSHOT or STATE_DELTA
    if (event.type === 'STATE_SNAPSHOT') {
      console.log('Full state received:', state);
    } else {
      console.log('State patches:', event.delta);
    }
    // Re-render with new state
    context.requestUpdate();
  }
});
```

### State-Only Tools (No UI)

Tools can react to state without rendering any UI:

```javascript
AguiTools.register('state_logger', {
  render: () => {}, // No-op render

  onStateEvent: ({ state, event }) => {
    // Log all state changes
    console.log('Agent state changed:', state);

    // Trigger side effects based on state
    if (state?.notification) {
      showNotification(state.notification);
    }
  }
});
```

### JSON Patch Operations

Common operations for `sendStateUpdate`:

```javascript
// Add a value
{ op: 'add', path: '/user/preferences', value: { theme: 'dark' } }

// Replace a value
{ op: 'replace', path: '/status', value: 'approved' }

// Remove a value
{ op: 'remove', path: '/temporary_data' }

// Add to end of array
{ op: 'add', path: '/items/-', value: { name: 'New Item' } }
```

## Sending Messages to the Agent

Tools can programmatically send messages to the agent using `context.sendMessage()`.
This triggers a new agent run, allowing tools to prompt the agent for more actions
based on user interactions.

### Basic Usage

```javascript
AguiTools.register('research_board', {
  render: ({ container, state, sendMessage }) => {
    container.innerHTML = `
      <div class="research-board">
        <h3>${state?.research?.topic}</h3>
        <button class="dig-deeper">Dig Deeper</button>
      </div>
    `;

    container.querySelector('.dig-deeper')?.addEventListener('click', () => {
      // Send a message that prompts the agent to continue research
      sendMessage('Please dig deeper into "' + state?.research?.topic + '". I\'d like more detailed research, additional sources, and more findings. Focus especially on any topics I\'ve starred.');
    });
  }
});
```

### Combining State Updates with Messages

A common pattern is to update the state first (so the agent sees user feedback),
then send a message to trigger the agent to act on that feedback:

```javascript
AguiTools.register('feedback_tool', {
  render: ({ container, state, sendStateUpdate, sendMessage }) => {
    container.innerHTML = `
      <div class="feedback">
        <button class="request-more">Get More Details</button>
        <button class="star-item" data-id="item-1">⭐ Star</button>
      </div>
    `;

    // Star an item, then ask agent to focus on starred items
    container.querySelector('.star-item')?.addEventListener('click', (e) => {
      const itemId = e.target.dataset.id;

      // First, update state so agent knows what's starred
      sendStateUpdate([
        { op: 'add', path: '/userFeedback/starredItems/-', value: itemId }
      ]);

      // Then prompt agent to act on the feedback
      sendMessage('I\'ve starred some items. Please provide more details on the starred items.');
    });

    // Simple request for more without state change
    container.querySelector('.request-more')?.addEventListener('click', () => {
      sendMessage('Please provide more details on this topic.');
    });
  }
});
```

### Use Cases

- **"Dig Deeper" buttons** - Allow users to request more research/analysis
- **Follow-up questions** - Let users ask the agent to elaborate on findings
- **Action triggers** - Request the agent to perform specific actions based on UI state
- **Feedback loops** - Update state with user preferences, then ask agent to adjust

### Important Notes

1. **Message triggers a new run**: Calling `sendMessage()` starts a new agent conversation turn, just like typing in the chat input.

2. **State is included**: The current state (including any updates from `sendStateUpdate`) is automatically included with the message, so the agent has full context.

3. **Avoid rapid calls**: Don't call `sendMessage()` in rapid succession - wait for the agent to complete before sending another message.

