/**
 * @file
 * Simple tool renderer example - JavaScript only.
 *
 * This demonstrates the simplest way to create a custom tool renderer.
 * Everything is in one file: HTML, CSS, and JS.
 *
 * Best for: Quick prototypes, simple tools, sharing code snippets.
 *
 * To use: Attach as a library in your render array:
 *   $build['#attached']['library'][] = 'my_module/simple_tool';
 *
 * @see ../README.md for comparison with SDC approach.
 */

(function () {
  'use strict';

  /**
   * Simple search tool renderer.
   *
   * Shows search progress and results in a basic list format.
   */
  const searchRenderer = {
    // Optional: CSS class for the container.
    containerClass: 'simple-search-widget',

    /**
     * Render function called for each tool state change.
     *
     * @param {Object} ctx - Render context.
     * @param {string} ctx.toolName - The tool name.
     * @param {Object} ctx.args - Tool arguments.
     * @param {string} ctx.status - Current status (pending|streaming|complete|error).
     * @param {*} ctx.result - Tool result when complete.
     * @param {HTMLElement} ctx.container - DOM element to render into.
     */
    render({ args, status, result, container }) {
      const query = args.query || args.search || '...';

      // Loading state.
      if (status === 'pending' || status === 'streaming') {
        container.innerHTML = `
          <div class="simple-search simple-search--loading">
            <span class="simple-search__icon">🔍</span>
            <span class="simple-search__text">Searching for "${escapeHtml(query)}"...</span>
          </div>
        `;
        return;
      }

      // Error state.
      if (status === 'error') {
        container.innerHTML = `
          <div class="simple-search simple-search--error">
            <span class="simple-search__icon">⚠️</span>
            <span class="simple-search__text">Search failed for "${escapeHtml(query)}"</span>
          </div>
        `;
        return;
      }

      // Parse results.
      let results = parseResults(result);

      // Empty state.
      if (results.length === 0) {
        container.innerHTML = `
          <div class="simple-search simple-search--empty">
            <span class="simple-search__icon">📭</span>
            <span class="simple-search__text">No results found for "${escapeHtml(query)}"</span>
          </div>
        `;
        return;
      }

      // Results state.
      container.innerHTML = `
        <div class="simple-search simple-search--complete">
          <div class="simple-search__header">
            Found ${results.length} result${results.length !== 1 ? 's' : ''} for "${escapeHtml(query)}"
          </div>
          <ul class="simple-search__list">
            ${results.map((r) => `
              <li class="simple-search__item">
                <a href="${escapeHtml(r.url || '#')}" class="simple-search__link">
                  ${escapeHtml(r.title || 'Untitled')}
                </a>
                ${r.snippet ? `<p class="simple-search__snippet">${escapeHtml(r.snippet)}</p>` : ''}
              </li>
            `).join('')}
          </ul>
        </div>
      `;
    },
  };

  /**
   * Parses tool result into an array of results.
   */
  function parseResults(result) {
    if (Array.isArray(result)) return result;
    if (typeof result === 'string') {
      try {
        const parsed = JSON.parse(result);
        return Array.isArray(parsed) ? parsed : [];
      } catch {
        return [];
      }
    }
    return [];
  }

  /**
   * Escapes HTML to prevent XSS.
   */
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * Injects minimal styles for the simple renderer.
   * In production, you'd put these in a separate CSS file.
   */
  function injectStyles() {
    if (document.getElementById('simple-search-styles')) return;

    const style = document.createElement('style');
    style.id = 'simple-search-styles';
    style.textContent = `
      .simple-search {
        padding: 0.75rem 1rem;
        border-radius: 8px;
        font-size: 0.9rem;
      }

      .simple-search--loading {
        background: #e3f2fd;
        color: #1565c0;
      }

      .simple-search--error {
        background: #ffebee;
        color: #c62828;
      }

      .simple-search--empty {
        background: #fff3e0;
        color: #e65100;
      }

      .simple-search--complete {
        background: #f5f5f5;
      }

      .simple-search__icon {
        margin-right: 0.5rem;
      }

      .simple-search__header {
        font-size: 0.8rem;
        color: #666;
        margin-bottom: 0.5rem;
      }

      .simple-search__list {
        list-style: none;
        padding: 0;
        margin: 0;
      }

      .simple-search__item {
        padding: 0.5rem 0;
        border-bottom: 1px solid #e0e0e0;
      }

      .simple-search__item:last-child {
        border-bottom: none;
      }

      .simple-search__link {
        color: #1976d2;
        text-decoration: none;
        font-weight: 500;
      }

      .simple-search__link:hover {
        text-decoration: underline;
      }

      .simple-search__snippet {
        margin: 0.25rem 0 0;
        font-size: 0.85rem;
        color: #666;
      }
    `;
    document.head.appendChild(style);
  }

  /**
   * Registers the tool renderer.
   */
  function register() {
    injectStyles();

    // Use the queue pattern (works before API loads).
    window.AguiToolsQueue = window.AguiToolsQueue || [];
    window.AguiToolsQueue.push({
      name: 'search_documents',
      config: searchRenderer,
    });

    // Also register directly if API is available.
    if (window.AguiTools) {
      window.AguiTools.register('search_documents', searchRenderer);
    }

    // And listen for the ready event (handles race conditions).
    window.addEventListener('agui:tools:ready', (event) => {
      event.detail.api.register('search_documents', searchRenderer);
    });
  }

  // Auto-register on load.
  register();

})();

