<?php

namespace Drupal\ai_404_redirect\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for AI 404 Redirect settings.
 */
class Ai404RedirectSettingsForm extends ConfigFormBase {

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(AiProviderPluginManager $ai_provider_manager) {
    $this->aiProviderManager = $ai_provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_404_redirect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_404_redirect_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_404_redirect.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable AI 404 Redirect'),
      '#description' => $this->t('When enabled, 404 errors will be analyzed by AI to suggest redirects.'),
      '#default_value' => $config->get('enabled'),
    ];

    // Get available AI providers.
    $provider_definitions = $this->aiProviderManager->getDefinitions();
    $provider_options = [];
    foreach ($provider_definitions as $provider_id => $definition) {
      $provider_options[$provider_id] = $definition['label'] ?? $provider_id;
    }

    $form['ai_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Provider'),
      '#description' => $this->t('Select the AI provider to use for analyzing 404 errors. Make sure the provider is configured with valid API keys at <a href="@url">/admin/config/ai/models</a>.', [
        '@url' => '/admin/config/ai/models',
      ]),
      '#options' => $provider_options,
      '#default_value' => $config->get('ai_provider') ?? 'openai',
      '#required' => TRUE,
    ];
    
    // Add a status message if provider is configured.
    $selected_provider = $config->get('ai_provider') ?? 'openai';
    if (isset($provider_options[$selected_provider])) {
      try {
        $provider = $this->aiProviderManager->createInstance($selected_provider);
        $is_usable = $provider->isUsable('chat');
        $models = $is_usable ? $provider->getConfiguredModels('chat') : [];
        
        if (!$is_usable || empty($models)) {
          $form['ai_provider_status'] = [
            '#type' => 'markup',
            '#markup' => '<div class="messages messages--warning">' . 
              $this->t('⚠️ The selected AI provider (@provider) is not properly configured. Please configure API keys and models at <a href="@url">/admin/config/ai/models</a>. The system will use fallback analysis until this is fixed.', [
                '@provider' => $provider_options[$selected_provider],
                '@url' => '/admin/config/ai/models',
              ]) . '</div>',
            '#weight' => -10,
          ];
        }
        else {
          $form['ai_provider_status'] = [
            '#type' => 'markup',
            '#markup' => '<div class="messages messages--status">' . 
              $this->t('✓ AI provider (@provider) is configured with @count model(s).', [
                '@provider' => $provider_options[$selected_provider],
                '@count' => count($models),
              ]) . '</div>',
            '#weight' => -10,
          ];
        }
      }
      catch (\Exception $e) {
        $form['ai_provider_status'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--error">' . 
            $this->t('Error checking AI provider: @message', [
              '@message' => $e->getMessage(),
            ]) . '</div>',
          '#weight' => -10,
        ];
      }
    }

    $form['confidence_thresholds'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Confidence Thresholds'),
      '#description' => $this->t('Set confidence score thresholds for different levels.'),
    ];

    $form['confidence_thresholds']['confidence_threshold_low'] = [
      '#type' => 'number',
      '#title' => $this->t('Low Confidence Threshold'),
      '#description' => $this->t('Minimum confidence score (0-100) for low confidence level.'),
      '#default_value' => $config->get('confidence_threshold_low') ?? 50,
      '#min' => 0,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['confidence_thresholds']['confidence_threshold_medium'] = [
      '#type' => 'number',
      '#title' => $this->t('Medium Confidence Threshold'),
      '#description' => $this->t('Minimum confidence score (0-100) for medium confidence level.'),
      '#default_value' => $config->get('confidence_threshold_medium') ?? 70,
      '#min' => 0,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['confidence_thresholds']['confidence_threshold_high'] = [
      '#type' => 'number',
      '#title' => $this->t('High Confidence Threshold'),
      '#description' => $this->t('Minimum confidence score (0-100) for high confidence level.'),
      '#default_value' => $config->get('confidence_threshold_high') ?? 85,
      '#min' => 0,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['auto_approve_confidence_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Auto-Approve Confidence Threshold'),
      '#description' => $this->t('Confidence score (0-100) above which redirects will be automatically created. Set to 100 to disable auto-approval.'),
      '#default_value' => $config->get('auto_approve_confidence_threshold') ?? 90,
      '#min' => 0,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['redirect_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Redirect Creation Settings'),
      '#description' => $this->t('Control when redirects are created.'),
    ];

    $form['redirect_settings']['minimum_404_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum 404 Count Before Redirect'),
      '#description' => $this->t('Number of times a 404 path must be encountered before creating a redirect. This prevents creating redirects for one-off typos. Set to 1 to create redirects immediately.'),
      '#default_value' => $config->get('minimum_404_count') ?? 4,
      '#min' => 1,
      '#max' => 100,
      '#required' => TRUE,
    ];

    $form['bot_blocking'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Bot & Trolling Protection'),
      '#description' => $this->t('Settings to prevent bots and trolling attempts from creating redirects.'),
    ];

    $form['bot_blocking']['enable_bot_blocking'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Bot/Trolling Blocking'),
      '#description' => $this->t('When enabled, suspicious activity (multiple 404s from same IP, exploitation attempts) will be blocked.'),
      '#default_value' => $config->get('enable_bot_blocking') ?? TRUE,
    ];

    $form['bot_blocking']['suspicious_ip_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Suspicious IP Threshold'),
      '#description' => $this->t('Number of unique 404 paths from the same IP address within 1 hour that triggers blocking. This helps prevent trolling bots.'),
      '#default_value' => $config->get('suspicious_ip_threshold') ?? 10,
      '#min' => 1,
      '#max' => 100,
      '#required' => TRUE,
      '#states' => [
        'enabled' => [
          ':input[name="enable_bot_blocking"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_404_redirect.settings');
    $config->set('enabled', $form_state->getValue('enabled'));
    $config->set('ai_provider', $form_state->getValue('ai_provider'));
    $config->set('confidence_threshold_low', $form_state->getValue('confidence_threshold_low'));
    $config->set('confidence_threshold_medium', $form_state->getValue('confidence_threshold_medium'));
    $config->set('confidence_threshold_high', $form_state->getValue('confidence_threshold_high'));
    $config->set('auto_approve_confidence_threshold', $form_state->getValue('auto_approve_confidence_threshold'));
    $config->set('minimum_404_count', $form_state->getValue('minimum_404_count'));
    $config->set('enable_bot_blocking', $form_state->getValue('enable_bot_blocking'));
    $config->set('suspicious_ip_threshold', $form_state->getValue('suspicious_ip_threshold'));
    $config->save();

    parent::submitForm($form, $form_state);
  }

}

