<?php

namespace Drupal\ai_404_redirect\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\ai_404_redirect\Service\AiRedirectAnalyzer;

/**
 * Event subscriber to handle 404 errors and analyze them with AI.
 */
class Ai404RedirectSubscriber implements EventSubscriberInterface {

  /**
   * The AI redirect analyzer service.
   *
   * @var \Drupal\ai_404_redirect\Service\AiRedirectAnalyzer
   */
  protected $analyzer;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\Core\Path\PathAliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new Ai404RedirectSubscriber.
   *
   * @param \Drupal\ai_404_redirect\Service\AiRedirectAnalyzer $analyzer
   *   The AI redirect analyzer service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    AiRedirectAnalyzer $analyzer,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    AliasManagerInterface $path_alias_manager,
    AccountProxyInterface $current_user
  ) {
    $this->analyzer = $analyzer;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->pathAliasManager = $path_alias_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::EXCEPTION][] = ['onException', 50];
    return $events;
  }

  /**
   * Handles 404 exceptions.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The exception event.
   */
  public function onException(ExceptionEvent $event) {
    // Symfony 5.4+ uses getThrowable() instead of getException().
    $throwable = method_exists($event, 'getThrowable') ? $event->getThrowable() : $event->getException();
    
    // Only handle 404 errors.
    if (!$throwable instanceof NotFoundHttpException) {
      return;
    }

    $request = $event->getRequest();
    $path = $request->getPathInfo();
    
    // Skip admin paths and system paths.
    if (strpos($path, '/admin') === 0 || 
        strpos($path, '/system') === 0 ||
        strpos($path, '/node') === 0 ||
        strpos($path, '/user') === 0) {
      return;
    }

    // Get configuration.
    $config = $this->configFactory->get('ai_404_redirect.settings');
    if (!$config->get('enabled')) {
      return;
    }

    // Queue the analysis instead of running it synchronously.
    // This prevents blocking the 404 response and improves performance.
    $queue = \Drupal::queue('ai_404_redirect_analysis');
    $queue->createItem([
      'path' => $path,
      'user_agent' => $request->headers->get('User-Agent', ''),
      'client_ip' => $request->getClientIp(),
      'timestamp' => \Drupal::time()->getRequestTime(),
    ]);
  }

}

