<?php

namespace Drupal\ai_404_redirect\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Approves a redirect suggestion and creates the redirect.
 *
 * @Action(
 *   id = "ai_404_redirect_approve",
 *   label = @Translation("Approve redirect"),
 *   type = "ai_404_redirect_suggestion"
 * )
 */
class ApproveRedirect extends ActionBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($suggestion = NULL) {
    if (!$suggestion || empty($suggestion->suggested_path)) {
      return;
    }

    $database = \Drupal::database();
    
    // Check if redirect already exists.
    $redirect_storage = $this->entityTypeManager->getStorage('redirect');
    $existing = $redirect_storage->loadByProperties([
      'redirect_source__path' => $suggestion->source_path,
    ]);

    if (!empty($existing)) {
      // Update status to approved.
      $database->update('ai_404_redirect_suggestions')
        ->fields(['status' => 'approved'])
        ->condition('id', $suggestion->id)
        ->execute();
      return;
    }

    // Create redirect.
    try {
      // Validate and normalize suggested path for URI.
      $suggested_path = $suggestion->suggested_path;
      if (empty($suggested_path)) {
        \Drupal::logger('ai_404_redirect')->error('Cannot create redirect: suggested_path is empty for suggestion ID: @id', [
          '@id' => $suggestion->id,
        ]);
        return;
      }
      
      // Remove 'internal:' prefix if present (setRedirect handles this).
      $suggested_path = preg_replace('/^internal:/', '', $suggested_path);
      
      // Ensure suggested path starts with / for internal paths.
      if (!str_starts_with($suggested_path, '/')) {
        $suggested_path = '/' . $suggested_path;
      }
      
      // Validate that the destination path exists (basic check).
      try {
        $destination_url = \Drupal::service('path.validator')->getUrlIfValid($suggested_path);
        if (!$destination_url) {
          \Drupal::logger('ai_404_redirect')->warning('Suggested path may not be valid: @path. Creating redirect anyway.', [
            '@path' => $suggested_path,
          ]);
        }
      }
      catch (\Exception $e) {
        // Path validator might throw exceptions, but we'll continue anyway.
      }
      
      // The Redirect module expects the source path WITHOUT leading slash.
      // Clean the path (remove query string and fragment).
      $source_path_clean = strtok($suggestion->source_path, '?#');
      $source_path_clean = ltrim($source_path_clean, '/');
      
      if (empty($source_path_clean)) {
        \Drupal::logger('ai_404_redirect')->error('Cannot create redirect: source path is empty after cleaning for suggestion ID: @id', [
          '@id' => $suggestion->id,
        ]);
        return;
      }
      
      // Create redirect entity.
      $redirect = $redirect_storage->create([
        'status_code' => 301,
        'language' => 'und',
      ]);
      
      // Set the source path properly - Redirect module expects no leading slash.
      $redirect->get('redirect_source')->set(0, ['path' => $source_path_clean]);
      
      // Set the redirect destination using the proper method.
      // The setRedirect() method handles URI formatting correctly.
      // It expects the path without 'internal:' prefix and handles it automatically.
      $redirect->setRedirect($suggested_path);
      
      // Verify the redirect URL was set correctly.
      $redirect_url = $redirect->getRedirectUrl();
      if (!$redirect_url) {
        \Drupal::logger('ai_404_redirect')->error('Failed to set redirect URL for suggestion ID @id: @source -> @dest', [
          '@id' => $suggestion->id,
          '@source' => $source_path_clean,
          '@dest' => $suggested_path,
        ]);
        return;
      }
      
      // Validate the redirect before saving.
      $violations = $redirect->validate();
      if ($violations->count() > 0) {
        $messages = [];
        foreach ($violations as $violation) {
          $messages[] = $violation->getMessage();
        }
        \Drupal::logger('ai_404_redirect')->error('Redirect validation failed for suggestion ID @id: @messages', [
          '@id' => $suggestion->id,
          '@messages' => implode(', ', $messages),
        ]);
        return;
      }
      
      $redirect->save();

      // Update suggestion.
      $database->update('ai_404_redirect_suggestions')
        ->fields([
          'status' => 'approved',
          'redirect_id' => $redirect->id(),
          'updated' => \Drupal::time()->getRequestTime(),
        ])
        ->condition('id', $suggestion->id)
        ->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_404_redirect')->error('Failed to approve redirect: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $account->hasPermission('administer ai 404 redirect');
  }

}


