<?php

namespace Drupal\ai_404_redirect\Plugin\views\query;

use Drupal\views\Plugin\views\query\QueryPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\views\ResultRow;

/**
 * Views query plugin for AI 404 Redirect suggestions.
 *
 * @ViewsQuery(
 *   id = "ai_404_redirect",
 *   title = @Translation("AI 404 Redirect"),
 *   help = @Translation("Query against the AI 404 Redirect suggestions table.")
 * )
 */
class Ai404RedirectQuery extends QueryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function build(ViewExecutable $view) {
    // Process sort handlers to ensure they're added to orderby.
    // This is called before execute() and ensures table column clicks work.
    $exposed_input = $view->getExposedInput();
    
    // Check for table column sort clicks (via URL parameters).
    // Table style uses 'order' parameter with field name.
    if (isset($exposed_input['order'])) {
      $sort_field = $exposed_input['order'];
      $sort_order = isset($exposed_input['sort']) ? $exposed_input['sort'] : 'ASC';
      
      // Map field names if needed.
      $field_mapping = [
        'count_404' => '404_count',
      ];
      $db_field = $field_mapping[$sort_field] ?? $sort_field;
      
      // Add the order by.
      $this->addOrderBy('ai_404_redirect_suggestions', $db_field, $sort_order);
    }
    
    // Also process configured sort handlers.
    if (!empty($view->sort)) {
      foreach ($view->sort as $sort_id => $sort_handler) {
        // Skip if this sort is already handled by exposed input.
        if (isset($exposed_input['order']) && $sort_handler->field === $exposed_input['order']) {
          continue;
        }
        
        if (!empty($sort_handler->field)) {
          $field_name = $sort_handler->field;
          $table = $sort_handler->table ?? 'ai_404_redirect_suggestions';
          
          // Get sort order from handler options.
          $order = strtoupper($sort_handler->options['order'] ?? 'ASC');
          
          // Add the order by.
          $this->addOrderBy($table, $field_name, $order);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function ensureTable($table, $relationship = NULL) {
    // For our simple table-based query, we always use the base table.
    return 's';
  }

  /**
   * Track fields that need to be added to the query.
   *
   * @var array
   */
  protected $fields = [];

  /**
   * Track where conditions grouped by group ID.
   *
   * @var array
   */
  protected $where = [];

  /**
   * Track order by clauses.
   *
   * @var array
   */
  protected $orderby = [];

  /**
   * {@inheritdoc}
   */
  public function addField($table, $field, $alias = '', $params = []) {
    // For our simple query, we just track the field.
    $alias = $alias ?: $field;
    $this->fields[$alias] = [
      'table' => $table,
      'field' => $field,
      'alias' => $alias,
      'params' => $params,
    ];
    return $alias;
  }

  /**
   * {@inheritdoc}
   */
  public function addWhere($group, $field, $value = NULL, $operator = NULL) {
    // Ensure the where array structure exists.
    if (!isset($this->where[$group])) {
      $this->where[$group] = ['conditions' => [], 'args' => []];
    }
    
    // Add the condition.
    $this->where[$group]['conditions'][] = [
      'field' => $field,
      'value' => $value,
      'operator' => $operator ?? '=',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function addOrderBy($table, $field = NULL, $order = 'ASC', $alias = '', $params = []) {
    // Only add if we have a field.
    if ($field) {
      // Handle different field formats:
      // - "field_name" (just the field)
      // - "s.field_name" (with table prefix)
      // - "table.field_name" (full table.field format)
      // - Field alias from Views
      
      // Remove table prefix if present.
      $field_name = $field;
      if (strpos($field, '.') !== FALSE) {
        $field_name = substr($field, strpos($field, '.') + 1);
      }
      
      // Map Views field aliases to database field names.
      $field_mapping = [
        'count_404' => '404_count',
      ];
      $db_field = $field_mapping[$field_name] ?? $field_name;
      
      // Store the order by with the database field name.
      $this->orderby[] = [
        'field' => $db_field,
        'direction' => strtoupper($order),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(ViewExecutable $view) {
    $database = \Drupal::database();
    $query = $database->select('ai_404_redirect_suggestions', 's');

    // Add fields that were requested via addField().
    // If no specific fields were requested, get all fields.
    if (empty($this->fields)) {
      $query->fields('s');
    } else {
      // Add only the fields that were requested.
      foreach ($this->fields as $alias => $field_info) {
        $query->addField('s', $field_info['field'], $alias);
      }
    }

    // Apply filters.
    foreach ($this->where as $group) {
      foreach ($group['conditions'] as $condition) {
        if (isset($condition['field'])) {
          $field = $condition['field'];
          $value = $condition['value'];
          $operator = $condition['operator'] ?? '=';
          
          // Remove table alias prefix if present (e.g., "s.field" -> "field").
          if (strpos($field, '.') !== FALSE) {
            $field = substr($field, strpos($field, '.') + 1);
          }
          
          // Map field names (e.g., count_404 -> 404_count).
          $field_mapping = [
            'count_404' => '404_count',
          ];
          $db_field = $field_mapping[$field] ?? $field;
          
          // Handle boolean values - convert string '0'/'1' to integer.
          if (is_string($value) && ($value === '0' || $value === '1')) {
            $value = (int) $value;
          }
          
          if ($operator === '=') {
            $query->condition("s.$db_field", $value);
          }
          elseif ($operator === 'IN') {
            $query->condition("s.$db_field", $value, 'IN');
          }
          elseif ($operator === '>=') {
            $query->condition("s.$db_field", $value, '>=');
          }
          elseif ($operator === '>') {
            $query->condition("s.$db_field", $value, '>');
          }
          elseif ($operator === '<') {
            $query->condition("s.$db_field", $value, '<');
          }
        }
      }
    }

    // Apply sorting from addOrderBy() calls (processed in build()).
    if (!empty($this->orderby)) {
      foreach ($this->orderby as $order) {
        if (!empty($order['field'])) {
          // The field name is already mapped in addOrderBy().
          $db_field = $order['field'];
          $query->orderBy("s.$db_field", $order['direction']);
        }
      }
    }

    // Apply paging.
    if (!empty($this->limit)) {
      $query->range($this->offset, $this->limit);
    }

    $results = $query->execute()->fetchAll(\PDO::FETCH_OBJ);

    $index = 0;
    foreach ($results as $row) {
      $result_row = new ResultRow();
      // Map all database fields to the result row.
      // Field handlers access fields by their alias (set via addField()).
      foreach ((array) $row as $key => $value) {
        // Set the field using the alias from the query.
        $result_row->{$key} = $value;
      }
      $result_row->index = $index++;
      // Initialize field data structure for field handlers.
      $result_row->_field_data = [];
      $view->result[] = $result_row;
    }
    
    // Set total rows for pager.
    $view->total_rows = count($view->result);
  }

}

