<?php

namespace Drupal\Tests\ai_404_redirect\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\node\Entity\Node;
use Drupal\path_alias\Entity\PathAlias;

/**
 * Functional tests for confidence scoring in real scenarios.
 *
 * @group ai_404_redirect
 */
class ConfidenceScoringFunctionalTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'path_alias',
    'ai_404_redirect',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    
    // Create test content.
    $this->createTestContent();
  }

  /**
   * Creates test content for testing.
   */
  protected function createTestContent() {
    // Create a page with alias /who-we-are/careers.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Careers',
      'status' => 1,
    ]);
    $node->save();
    
    PathAlias::create([
      'path' => '/node/' . $node->id(),
      'alias' => '/who-we-are/careers',
      'langcode' => 'en',
    ])->save();
    
    // Create a page with alias /what-we-offer.
    $node2 = Node::create([
      'type' => 'page',
      'title' => 'What We Offer',
      'status' => 1,
    ]);
    $node2->save();
    
    PathAlias::create([
      'path' => '/node/' . $node2->id(),
      'alias' => '/what-we-offer',
      'langcode' => 'en',
    ])->save();
    
    // Rebuild alias index.
    require_once DRUPAL_ROOT . '/modules/custom/ai_404_redirect/ai_404_redirect.install';
    ai_404_redirect_rebuild_alias_index();
  }

  /**
   * Test confidence for typo in URL.
   */
  public function testTypoConfidence() {
    // Visit a typo URL: /who-we-were/careers (should match /who-we-are/careers).
    $this->drupalGet('/who-we-were/careers');
    
    // Wait for queue processing.
    $this->container->get('cron')->run();
    
    // Check suggestion was created with good confidence.
    $suggestion = \Drupal::database()
      ->select('ai_404_redirect_suggestions', 's')
      ->fields('s', ['confidence_score', 'suggested_path', 'status'])
      ->condition('source_path', '/who-we-were/careers')
      ->orderBy('created', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchObject();
    
    if ($suggestion) {
      $this->assertGreaterThan(50, $suggestion->confidence_score,
        'Typo match should have confidence > 50');
      $this->assertEquals('/who-we-are/careers', $suggestion->suggested_path,
        'Should suggest correct path');
    }
  }

  /**
   * Test confidence for exact match.
   */
  public function testExactMatchConfidence() {
    // Visit close match: /what-we-offered (should match /what-we-offer).
    $this->drupalGet('/what-we-offered');
    
    $this->container->get('cron')->run();
    
    $suggestion = \Drupal::database()
      ->select('ai_404_redirect_suggestions', 's')
      ->fields('s', ['confidence_score', 'suggested_path'])
      ->condition('source_path', '/what-we-offered')
      ->orderBy('created', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchObject();
    
    if ($suggestion) {
      $this->assertGreaterThan(60, $suggestion->confidence_score,
        'Close match should have confidence > 60');
      $this->assertEquals('/what-we-offer', $suggestion->suggested_path,
        'Should suggest correct path');
    }
  }

  /**
   * Test auto-approval based on confidence threshold.
   */
  public function testAutoApprovalThreshold() {
    $config = \Drupal::configFactory()->getEditable('ai_404_redirect.settings');
    $threshold = $config->get('auto_approve_confidence_threshold') ?? 80;
    
    // Visit a high-confidence match.
    $this->drupalGet('/who-we-are/careers-page');
    
    $this->container->get('cron')->run();
    
    $suggestion = \Drupal::database()
      ->select('ai_404_redirect_suggestions', 's')
      ->fields('s', ['confidence_score', 'status'])
      ->condition('source_path', '/who-we-are/careers-page')
      ->orderBy('created', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchObject();
    
    if ($suggestion) {
      if ($suggestion->confidence_score >= $threshold) {
        $this->assertEquals('auto_approved', $suggestion->status,
          'High confidence should auto-approve');
      } else {
        $this->assertEquals('pending', $suggestion->status,
          'Low confidence should remain pending');
      }
    }
  }

}
