<?php

namespace Drupal\Tests\ai_404_redirect\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\ai_404_redirect\Service\AiRedirectAnalyzer;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests for confidence scoring in AI 404 Redirect module.
 *
 * @group ai_404_redirect
 */
class ConfidenceScoringTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'path_alias',
    'ai_404_redirect',
  ];

  /**
   * The AI redirect analyzer service.
   *
   * @var \Drupal\ai_404_redirect\Service\AiRedirectAnalyzer
   */
  protected $analyzer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    
    $this->installSchema('system', ['sequences']);
    $this->installSchema('ai_404_redirect', ['ai_404_redirect_suggestions']);
    $this->installSchema('ai_404_redirect', ['ai_404_redirect_alias_index']);
    
    // Create mock services.
    $config_factory = $this->container->get('config.factory');
    $entity_type_manager = $this->container->get('entity_type.manager');
    $path_alias_manager = $this->container->get('path_alias.manager');
    $state = $this->container->get('state');
    
    // Mock AI provider manager (we'll test fallback analysis which doesn't need AI).
    $ai_provider_manager = $this->createMock(AiProviderPluginManager::class);
    
    // Create the analyzer service.
    $this->analyzer = new AiRedirectAnalyzer(
      $ai_provider_manager,
      $entity_type_manager,
      $path_alias_manager,
      $config_factory,
      $state
    );
    
    // Set configuration.
    $config = $config_factory->getEditable('ai_404_redirect.settings');
    $config->set('enabled', TRUE);
    $config->set('auto_approve_confidence_threshold', 80);
    $config->save();
  }

  /**
   * Test confidence scoring for exact path matches.
   */
  public function testExactPathMatchConfidence() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Land Loans',
        'alias' => '/loans/land',
        'type' => 'page',
      ],
    ];
    
    // Test exact match: /loans/land-01 should match /loans/land with high confidence.
    $path = '/loans/land-01';
    $is_bot = FALSE;
    
    // Use reflection to call protected method.
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Should have high confidence for exact path structure match.
    $this->assertGreaterThan(50, $result['confidence_score'], 
      'Exact path match should have confidence > 50');
    $this->assertTrue($result['should_redirect'], 
      'Exact path match should suggest redirect');
    $this->assertEquals('/loans/land', $result['suggested_path'],
      'Should suggest the exact matching path');
  }

  /**
   * Test confidence scoring for last segment matches.
   */
  public function testLastSegmentMatchConfidence() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Careers',
        'alias' => '/who-we-are/careers',
        'type' => 'page',
      ],
      [
        'nid' => 2,
        'title' => 'Financial Reports',
        'alias' => '/who-we-are/financial-reports',
        'type' => 'page',
      ],
    ];
    
    // Test: /who-we-were/careers should match /who-we-are/careers (last segment match).
    $path = '/who-we-were/careers';
    $is_bot = FALSE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Should prefer /who-we-are/careers over /who-we-are/financial-reports.
    $this->assertEquals('/who-we-are/careers', $result['suggested_path'],
      'Should match based on last segment (careers)');
    $this->assertGreaterThan(70, $result['confidence_score'],
      'Last segment match should have high confidence (>70)');
    $this->assertTrue($result['should_redirect'],
      'Last segment match should suggest redirect');
  }

  /**
   * Test confidence scoring for typo matches.
   */
  public function testTypoMatchConfidence() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'What We Offer',
        'alias' => '/what-we-offer',
        'type' => 'page',
      ],
    ];
    
    // Test typos: offed, offfer should match offer.
    $typos = ['/what-we-offed', '/what-we-offfer'];
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    foreach ($typos as $typo_path) {
      $result = $method->invoke($this->analyzer, $typo_path, $candidates, FALSE);
      
      $this->assertEquals('/what-we-offer', $result['suggested_path'],
        "Typo '$typo_path' should match '/what-we-offer'");
      $this->assertGreaterThan(40, $result['confidence_score'],
        "Typo match should have reasonable confidence (>40)");
    }
  }

  /**
   * Test confidence scoring for low-quality matches.
   */
  public function testLowConfidenceMatches() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Home Page',
        'alias' => '/home',
        'type' => 'page',
      ],
    ];
    
    // Test completely unrelated path.
    $path = '/random-gibberish-xyz123';
    $is_bot = FALSE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Should have low confidence or no redirect suggestion.
    if ($result['should_redirect']) {
      $this->assertLessThan(50, $result['confidence_score'],
        'Unrelated paths should have low confidence (<50)');
    } else {
      $this->assertFalse($result['should_redirect'],
        'Completely unrelated paths should not suggest redirect');
    }
  }

  /**
   * Test auto-approval threshold.
   */
  public function testAutoApprovalThreshold() {
    $config = $this->container->get('config.factory')->getEditable('ai_404_redirect.settings');
    $threshold = $config->get('auto_approve_confidence_threshold') ?? 80;
    
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Land Loans',
        'alias' => '/loans/land',
        'type' => 'page',
      ],
    ];
    
    // Test high confidence match (should auto-approve).
    $path = '/loans/land-loan';
    $is_bot = FALSE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Check if confidence meets threshold.
    if ($result['confidence_score'] >= $threshold) {
      $this->assertTrue($result['should_redirect'],
        'High confidence matches should suggest redirect');
      
      // Test saveSuggestion to verify auto-approval.
      $save_method = $reflection->getMethod('saveSuggestion');
      $save_method->setAccessible(TRUE);
      $save_method->invoke($this->analyzer, $path, $result, 'Test User Agent', $is_bot);
      
      // Check database for auto_approved status.
      $suggestion = \Drupal::database()
        ->select('ai_404_redirect_suggestions', 's')
        ->fields('s', ['status', 'confidence_score'])
        ->condition('source_path', $path)
        ->orderBy('created', 'DESC')
        ->range(0, 1)
        ->execute()
        ->fetchObject();
      
      if ($suggestion && $suggestion->confidence_score >= $threshold) {
        $this->assertEquals('auto_approved', $suggestion->status,
          'High confidence suggestions should be auto-approved');
      }
    }
  }

  /**
   * Test confidence scaling and capping.
   */
  public function testConfidenceScaling() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Perfect Match Page',
        'alias' => '/perfect/match',
        'type' => 'page',
      ],
    ];
    
    // Test perfect match.
    $path = '/perfect/match';
    $is_bot = FALSE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Confidence should be capped at 100.
    $this->assertLessThanOrEqual(100, $result['confidence_score'],
      'Confidence should not exceed 100');
    $this->assertGreaterThanOrEqual(0, $result['confidence_score'],
      'Confidence should not be negative');
  }

  /**
   * Test bot requests have zero confidence.
   */
  public function testBotRequestConfidence() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Test Page',
        'alias' => '/test',
        'type' => 'page',
      ],
    ];
    
    $path = '/test-page';
    $is_bot = TRUE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Bot requests should not suggest redirects.
    $this->assertFalse($result['should_redirect'],
      'Bot requests should not suggest redirects');
    $this->assertStringContainsString('Bot', $result['reasoning'],
      'Bot requests should have bot-related reasoning');
  }

  /**
   * Test path structure scoring.
   */
  public function testPathStructureScoring() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('calculatePathStructureScore');
    $method->setAccessible(TRUE);
    
    // Test exact match in same position.
    $path1_parts = ['who', 'we', 'are', 'careers'];
    $path2_parts = ['who', 'we', 'are', 'careers'];
    $score = $method->invoke($this->analyzer, $path1_parts, $path2_parts);
    $this->assertGreaterThan(0.8, $score,
      'Exact path match should have high structure score (>0.8)');
    
    // Test partial match.
    $path1_parts = ['who', 'we', 'were', 'careers'];
    $path2_parts = ['who', 'we', 'are', 'careers'];
    $score = $method->invoke($this->analyzer, $path1_parts, $path2_parts);
    $this->assertGreaterThan(0.5, $score,
      'Similar path structure should have moderate score (>0.5)');
    
    // Test no match.
    $path1_parts = ['random', 'path'];
    $path2_parts = ['completely', 'different', 'path'];
    $score = $method->invoke($this->analyzer, $path1_parts, $path2_parts);
    $this->assertLessThan(0.3, $score,
      'Unrelated paths should have low structure score (<0.3)');
  }

  /**
   * Test confidence for multiple candidate matches.
   */
  public function testMultipleCandidateConfidence() {
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Land Loans',
        'alias' => '/loans/land',
        'type' => 'page',
      ],
      [
        'nid' => 2,
        'title' => 'Home Loans',
        'alias' => '/loans/home',
        'type' => 'page',
      ],
      [
        'nid' => 3,
        'title' => 'Auto Loans',
        'alias' => '/loans/auto',
        'type' => 'page',
      ],
    ];
    
    // Test: /loans/land-01 should match /loans/land (best match).
    $path = '/loans/land-01';
    $is_bot = FALSE;
    
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $result = $method->invoke($this->analyzer, $path, $candidates, $is_bot);
    
    // Should select the best match.
    $this->assertEquals('/loans/land', $result['suggested_path'],
      'Should select best matching candidate');
    $this->assertGreaterThan(50, $result['confidence_score'],
      'Best match should have reasonable confidence');
  }

}
