<?php

namespace Drupal\ai_404_redirect\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;

/**
 * Defines a bulk operations form element for AI 404 Redirect suggestions.
 *
 * @ViewsField("ai_404_redirect_bulk_form")
 */
class Ai404RedirectBulkForm extends FieldPluginBase implements TrustedCallbackInterface {

  /**
   * Gets available bulk action options.
   *
   * @return array
   *   Array of action IDs keyed by label.
   */
  protected function getBulkOptions() {
    $options = [];
    
    // Get available actions.
    $action_manager = \Drupal::service('plugin.manager.action');
    $definitions = $action_manager->getDefinitions();
    
    // Filter to only our actions.
    foreach ($definitions as $id => $definition) {
      if (strpos($id, 'ai_404_redirect_') === 0) {
        $options[$id] = $definition['label'];
      }
    }
    
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function viewsForm(&$form, FormStateInterface $form_state) {
    // Make sure we do not accidentally cache this form.
    $form['#cache']['max-age'] = 0;

    // Add the tableselect javascript.
    $form['#attached']['library'][] = 'core/drupal.tableselect';

    // Only add the bulk form options and buttons if there are results.
    if (!empty($this->view->result)) {
      // Add the checkboxes.
      $form[$this->options['id']]['#tree'] = TRUE;
      foreach ($this->view->result as $row_index => $row) {
        $form[$this->options['id']][$row_index] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Update this item'),
          '#title_display' => 'invisible',
          '#default_value' => !empty($form_state->getValue($this->options['id'])[$row_index]) ? 1 : NULL,
          '#return_value' => $row->id ?? $row_index,
        ];
      }

      // Replace the form submit button label.
      if (isset($form['actions']['submit'])) {
        $form['actions']['submit']['#value'] = $this->t('Apply to selected items');
      }

      // Ensure a consistent container for filters/operations in the view header.
      $form['header'] = [
        '#type' => 'container',
        '#weight' => -100,
      ];

      // Build the bulk operations action widget for the header.
      $action_title = $this->options['action_title'] ?? 'With selection';
      $form['header'][$this->options['id']] = [
        '#type' => 'container',
      ];
      
      $options = $this->getBulkOptions();
      if (!empty($options)) {
        $form['header'][$this->options['id']]['action'] = [
          '#type' => 'select',
          '#title' => $action_title,
          '#options' => $this->getBulkOptions(),
          '#empty_option' => $this->t('- Select -'),
        ];

        // Duplicate the form actions into the action container in the header.
        if (isset($form['actions'])) {
          $form['header'][$this->options['id']]['actions'] = $form['actions'];
        }
      }

      // Override the pre_render callback to use renderInIsolation for form elements.
      // This fixes the issue where placeholders aren't replaced because render()
      // requires a render context that may not be available.
      // We need to run our callback FIRST, before the core one, so we can replace
      // placeholders before the core callback tries (and fails) to replace them.
      if (!isset($form['#pre_render'])) {
        $form['#pre_render'] = [];
      }
      // Remove the core callback and add ours first.
      $form['#pre_render'] = array_filter($form['#pre_render'], function($callback) {
        if (is_array($callback) && isset($callback[0]) && is_string($callback[0])) {
          return $callback[0] !== 'Drupal\views\Form\ViewsFormMainForm' || $callback[1] !== 'preRenderViewsForm';
        }
        return TRUE;
      });
      // Add our callback first, then re-add the core one for header rendering.
      array_unshift($form['#pre_render'], [static::class, 'preRenderViewsFormFix']);
      $form['#pre_render'][] = ['Drupal\views\Form\ViewsFormMainForm', 'preRenderViewsForm'];
    }
    else {
      // Remove the default actions build array if no results.
      unset($form['actions']);
    }
  }

  /**
   * Pre-render callback to fix placeholder replacement.
   *
   * This runs after the core preRenderViewsForm and fixes any placeholders that
   * weren't replaced due to render context issues. Uses renderInIsolation()
   * instead of render() to avoid render context problems.
   *
   * @param array $element
   *   The form element.
   *
   * @return array
   *   The modified form element.
   */
  public static function preRenderViewsFormFix(array $element) {
    // Only process if we have substitutions and output.
    if (!isset($element['#substitutions']['#value']) || !isset($element['output'])) {
      return $element;
    }

    $renderer = \Drupal::service('renderer');
    
    // Get the current output - check if it's already been processed (has #markup).
    // If not, render it to get the HTML with placeholders.
    // Note: We need to render the output ourselves to get the placeholders,
    // as the core callback may have already processed it but failed to replace them.
    if (isset($element['output']['#markup'])) {
      // Already processed by core callback, but placeholders may not have been replaced.
      $rendered_output = is_string($element['output']['#markup']) 
        ? $element['output']['#markup'] 
        : (string) $element['output']['#markup'];
    }
    else {
      // Output is still a render array, render it to get HTML with placeholders.
      // Use renderRoot to ensure proper context.
      $rendered_output = $renderer->renderRoot($element['output']);
    }
    
    // Check if there are any placeholders in the output (both escaped and unescaped).
    $has_placeholders = FALSE;
    foreach ($element['#substitutions']['#value'] as $substitution) {
      $escaped_placeholder = \Drupal\Component\Utility\Html::escape($substitution['placeholder']);
      if (strpos($rendered_output, $substitution['placeholder']) !== FALSE || 
          strpos($rendered_output, $escaped_placeholder) !== FALSE) {
        $has_placeholders = TRUE;
        break;
      }
    }

    // Do replacement if placeholders are present.
    if ($has_placeholders) {
      $search = [];
      $replace = [];

      // Add in substitutions provided by the form.
      foreach ($element['#substitutions']['#value'] as $substitution) {
        $field_name = $substitution['field_name'];
        $row_id = $substitution['row_id'];
        $escaped_placeholder = \Drupal\Component\Utility\Html::escape($substitution['placeholder']);

        // Check for both escaped and unescaped versions.
        $placeholder_found = FALSE;
        $placeholder_to_replace = NULL;
        
        if (strpos($rendered_output, $substitution['placeholder']) !== FALSE) {
          $placeholder_found = TRUE;
          $placeholder_to_replace = $substitution['placeholder'];
        }
        elseif (strpos($rendered_output, $escaped_placeholder) !== FALSE) {
          $placeholder_found = TRUE;
          $placeholder_to_replace = $escaped_placeholder;
        }

        if ($placeholder_found) {
          $search[] = $placeholder_to_replace;
          // Use renderInIsolation() instead of render() to avoid render context issues.
          $replace[] = isset($element[$field_name][$row_id]) 
            ? $renderer->renderInIsolation($element[$field_name][$row_id]) 
            : '';
        }
      }

      // Apply substitutions to the rendered output.
      if (!empty($search)) {
        $output = str_replace($search, $replace, $rendered_output);
        $element['output'] = ['#markup' => \Drupal\views\Render\ViewsRenderPipelineMarkup::create($output)];
      }
    }
    // If no placeholders but output is still a render array, render it now.
    elseif (!isset($element['output']['#markup'])) {
      $rendered_output = $renderer->renderPlain($element['output']);
      $element['output'] = ['#markup' => \Drupal\views\Render\ViewsRenderPipelineMarkup::create($rendered_output)];
    }

    return $element;
  }

  /**
   * Submit handler for the bulk form.
   */
  public function submitBulkForm(array &$form, FormStateInterface $form_state) {
    // Get action from header container.
    $action_id = $form_state->getValue(['header', $this->options['id'], 'action']);
    if (empty($action_id)) {
      return;
    }

    $selected = [];
    $field_name = $this->options['id'];
    $selected_values = $form_state->getValue($field_name, []);
    
    foreach ($this->view->result as $row_index => $row) {
      if (!empty($selected_values[$row_index])) {
        $selected[] = $selected_values[$row_index];
      }
    }

    if (empty($selected)) {
      \Drupal::messenger()->addWarning($this->t('No items selected.'));
      return;
    }

    // Load the action plugin.
    $action_manager = \Drupal::service('plugin.manager.action');
    $action = $action_manager->createInstance($action_id);
    
    if (!$action) {
      \Drupal::messenger()->addError($this->t('Action not found.'));
      return;
    }

    // Execute action on each selected suggestion.
    $database = \Drupal::database();
    $processed = 0;
    
    foreach ($selected as $suggestion_id) {
      // suggestion_id might be the return_value from checkbox, which is the row ID
      $suggestion = $database->select('ai_404_redirect_suggestions', 's')
        ->fields('s')
        ->condition('s.id', $suggestion_id)
        ->execute()
        ->fetchObject();
      
      if ($suggestion) {
        try {
          $action->execute($suggestion);
          $processed++;
        }
        catch (\Exception $e) {
          \Drupal::logger('ai_404_redirect')->error('Error executing action on suggestion @id: @message', [
            '@id' => $suggestion_id,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    if ($processed > 0) {
      \Drupal::messenger()->addStatus($this->t('Applied action to @count item(s).', [
        '@count' => $processed,
      ]));
      
      // Invalidate cache.
      \Drupal::service('cache_tags.invalidator')->invalidateTags([
        'ai_404_redirect_suggestions',
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    // Return placeholder that will be replaced with the checkbox form element.
    return '<!--form-item-' . $this->options['id'] . '--' . $values->index . '-->';
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ResultRow $row, $field = NULL) {
    // Return placeholder that will be replaced with the checkbox form element.
    return '<!--form-item-' . $this->options['id'] . '--' . $row->index . '-->';
  }

  /**
   * {@inheritdoc}
   */
  public function preRender(&$values) {
    parent::preRender($values);

    // If the view is using a table style, add the tableselect CSS classes.
    if (!empty($this->view->style_plugin) && $this->view->style_plugin instanceof \Drupal\views\Plugin\views\style\Table) {
      // Add the tableselect css classes.
      $this->options['element_label_class'] .= ' select-all';
      // Hide the actual label of the field on the table header.
      $this->options['label'] = '';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // This field doesn't need to query anything.
  }

  /**
   * Returns the form element name for this field.
   *
   * @return string
   *   The form element name.
   */
  public function form_element_name() {
    return $this->options['id'];
  }

  /**
   * Returns the form element row ID for a given row index.
   *
   * @param int $row_index
   *   The row index.
   *
   * @return int
   *   The form element row ID.
   */
  public function form_element_row_id($row_index) {
    return $row_index;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderViewsFormFix'];
  }

}

