<?php

namespace Drupal\Tests\ai_404_redirect\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\ai_404_redirect\Service\AiRedirectAnalyzer;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Config\Config;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * Unit tests for confidence scoring calculations.
 *
 * @group ai_404_redirect
 */
class ConfidenceScoringUnitTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The analyzer service.
   *
   * @var \Drupal\ai_404_redirect\Service\AiRedirectAnalyzer
   */
  protected $analyzer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    
    // Create mock dependencies.
    $ai_provider_manager = $this->prophesize(AiProviderPluginManager::class)->reveal();
    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class)->reveal();
    $path_alias_manager = $this->prophesize(AliasManagerInterface::class)->reveal();
    
    $config = $this->prophesize(Config::class);
    $config->get('enabled')->willReturn(TRUE);
    $config->get('auto_approve_confidence_threshold')->willReturn(80);
    
    $config_factory = $this->prophesize(ConfigFactoryInterface::class);
    $config_factory->get('ai_404_redirect.settings')->willReturn($config->reveal());
    $config_factory->getEditable('ai_404_redirect.settings')->willReturn($config->reveal());
    
    $state = $this->prophesize(StateInterface::class);
    $state->get('ai_404_redirect.ai_disabled_reason')->willReturn(NULL);
    $state->get('ai_404_redirect.last_ai_call', 0)->willReturn(0);
    
    $this->analyzer = new AiRedirectAnalyzer(
      $ai_provider_manager,
      $entity_type_manager,
      $path_alias_manager,
      $config_factory->reveal(),
      $state->reveal()
    );
  }

  /**
   * Test calculatePathStructureScore method.
   */
  public function testCalculatePathStructureScore() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('calculatePathStructureScore');
    $method->setAccessible(TRUE);
    
    // Test 1: Exact match.
    $score = $method->invoke($this->analyzer, 
      ['who', 'we', 'are', 'careers'],
      ['who', 'we', 'are', 'careers']
    );
    $this->assertGreaterThan(0.8, $score, 'Exact match should score > 0.8');
    
    // Test 2: Last segment match (high priority).
    $score = $method->invoke($this->analyzer,
      ['who', 'we', 'were', 'careers'],
      ['who', 'we', 'are', 'careers']
    );
    $this->assertGreaterThan(0.6, $score, 'Last segment match should score > 0.6');
    
    // Test 3: No match.
    $score = $method->invoke($this->analyzer,
      ['random', 'path'],
      ['completely', 'different']
    );
    $this->assertLessThan(0.3, $score, 'No match should score < 0.3');
  }

  /**
   * Test calculateTypoSimilarity method.
   */
  public function testCalculateTypoSimilarity() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('calculateTypoSimilarity');
    $method->setAccessible(TRUE);
    
    // Test 1: Exact match.
    $similarity = $method->invoke($this->analyzer, 'offer', 'offer');
    $this->assertEquals(1.0, $similarity, 'Exact match should be 1.0');
    
    // Test 2: Common typo (offed -> offer).
    $similarity = $method->invoke($this->analyzer, 'offed', 'offer');
    $this->assertGreaterThan(0.7, $similarity, 'Typo should have similarity > 0.7');
    
    // Test 3: Different words.
    $similarity = $method->invoke($this->analyzer, 'offer', 'careers');
    $this->assertEquals(0, $similarity, 'Different words should be 0');
  }

  /**
   * Test generateFuzzyVariations method.
   */
  public function testGenerateFuzzyVariations() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('generateFuzzyVariations');
    $method->setAccessible(TRUE);
    
    // Test: "offed" should generate variations that could match "offer".
    $variations = $method->invoke($this->analyzer, 'offed');
    $this->assertNotEmpty($variations, 'Should generate fuzzy variations');
    
    // Check if variations include base form.
    $has_base = FALSE;
    foreach ($variations as $variation) {
      if (strpos($variation, 'off') === 0 && strlen($variation) <= 6) {
        $has_base = TRUE;
        break;
      }
    }
    $this->assertTrue($has_base, 'Should include base form variations');
  }

  /**
   * Test confidence score calculation and scaling.
   */
  public function testConfidenceScaling() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Test Page',
        'alias' => '/test/page',
        'type' => 'page',
      ],
    ];
    
    $result = $method->invoke($this->analyzer, '/test/page', $candidates, FALSE);
    
    // Confidence should be between 0 and 100.
    $this->assertGreaterThanOrEqual(0, $result['confidence_score']);
    $this->assertLessThanOrEqual(100, $result['confidence_score']);
  }

  /**
   * Test minimum score threshold for redirects.
   */
  public function testMinimumScoreThreshold() {
    $reflection = new \ReflectionClass($this->analyzer);
    $method = $reflection->getMethod('fallbackAnalysis');
    $method->setAccessible(TRUE);
    
    // Very poor match.
    $candidates = [
      [
        'nid' => 1,
        'title' => 'Unrelated Page',
        'alias' => '/completely/different',
        'type' => 'page',
      ],
    ];
    
    $result = $method->invoke($this->analyzer, '/random/gibberish', $candidates, FALSE);
    
    // Should not suggest redirect if score is too low.
    if ($result['confidence_score'] < 20) {
      $this->assertFalse($result['should_redirect'],
        'Very low confidence should not suggest redirect');
    }
  }

}
