<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:add_tool',
  label: new TranslatableMarkup('Add Agent Tool'),
  description: new TranslatableMarkup('This can add one agent tool from a specific agent by agent ID.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to add a tool to."),
      required: TRUE,
    ),
    'tool_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool ID"),
      description: new TranslatableMarkup("The machine name of the tool you want to addd from the agent."),
      required: TRUE,
    ),
    'return_directly' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Return Directly"),
      description: new TranslatableMarkup("If set to true the agent will stop processing and instead return the tool output directly. Default is false."),
      required: FALSE,
    ),
    'require_usage' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Require Usage"),
      description: new TranslatableMarkup("If set to true the agent will be required to use this tool once at least during its execution. Only set when the task for an agent is clear. If an agent can be used for answering question, a modification tool should not be set to true. Default is false."),
      required: FALSE,
    ),
    'tool_description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Description"),
      description: new TranslatableMarkup("An optional override description for the tool when added to the agent. If left empty, the default tool description will be used. Defaults to empty."),
      required: FALSE,
    ),
  ],
)]
class AddAgentTool extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The tool plugin manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $toolManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->toolManager = $container->get('plugin.manager.ai.function_calls');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'tool_id' => $tool_id,
      'return_directly' => $return_directly,
      'require_usage' => $require_usage,
      'tool_description' => $tool_description,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load the agent.
    /** @var \Drupal\ai_agents\AiAgentInterface $agent */
    $agent = $agent_storage->load($agent_id);
    if (!$agent) {
      return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
    }

    // Check if the tool exists.
    $tool_definition = $this->toolManager->getDefinition($tool_id);
    if (!$tool_definition) {
      return ExecutableResult::failure($this->t('Tool with ID %tool_id not found and can not be added to agent %agent_id.', [
        '%tool_id' => $tool_id,
        '%agent_id' => $agent_id,
      ]));
    }

    // Check if the tool already exists on the agent.
    $tools = $agent->get('tools');
    if (isset($tools[$tool_id])) {
      return ExecutableResult::failure($this->t('Tool with ID %tool_id already exists on agent %agent_id. Please remove first.', [
        '%tool_id' => $tool_id,
        '%agent_id' => $agent_id,
      ]));
    }

    // Add the tool to the agent.
    $tools[$tool_id] = TRUE;
    $agent->set('tools', $tools);

    // Add tool settings.
    $tool_settings = $agent->get('tool_settings');
    $tool_settings[$tool_id] = [
      'return_directly' => !empty($return_directly),
      'require_usage' => !empty($require_usage),
      'description_override' => $tool_description ?: '',
    ];

    $agent->set('tool_settings', $tool_settings);

    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent after adding tool: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t('Tool with ID %tool_id has been added to agent %agent_id.', [
      '%tool_id' => $tool_id,
      '%agent_id' => $agent_id,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
