<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:add_default_information_tool',
  label: new TranslatableMarkup('Add Default Information Tool'),
  description: new TranslatableMarkup('This can add one default information tool to a specific agent by agent ID.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to add a tool to."),
      required: TRUE,
    ),
    'tool_key' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Key"),
      description: new TranslatableMarkup("The key of the default information tool you want to add to the agent. Not the tool id."),
      required: TRUE,
    ),
    'tool_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool ID"),
      description: new TranslatableMarkup("The machine name of the tool you want to add to the agent."),
      required: TRUE,
    ),
    'tool_name' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Name"),
      description: new TranslatableMarkup("The human readable name of the tool you want to add to the agent."),
      required: TRUE,
    ),
    'tool_description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Description"),
      description: new TranslatableMarkup("The description of the tool you want to add to the agent."),
      required: TRUE,
    ),
    'tool_parameters' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Parameters"),
      description: new TranslatableMarkup('The parameters for the tool you want to add to the agent. Add them as a json string, e.g. {"param1":"value1","param2":"value2"}'),
      required: FALSE,
    ),
  ],
)]
class AddDefaultInformationTool extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The tool plugin manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $toolManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->toolManager = $container->get('plugin.manager.ai.function_calls');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'tool_key' => $tool_key,
      'tool_id' => $tool_id,
      'tool_name' => $tool_name,
      'tool_description' => $tool_description,
      'tool_parameters' => $tool_parameters,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load the agent.
    /** @var \Drupal\ai_agents\AiAgentInterface $agent */
    $agent = $agent_storage->load($agent_id);
    if (!$agent) {
      return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
    }

    // Check that the tool exists.
    $tool_definition = $this->toolManager->getDefinition($tool_id);
    if (!$tool_definition) {
      return ExecutableResult::failure($this->t('Tool with ID %id not found and can not be added to agent %agent_id.', [
        '%id' => $tool_id,
        '%agent_id' => $agent_id,
      ]));
    }

    // Check if the tool exists on the agent.
    $tools = $agent->get('default_information_tools');
    // Yaml decode it.
    $yaml = Yaml::parse($tools);

    $parameters = Json::decode($tool_parameters ?? '{}');
    foreach ($parameters as $key => $value) {
      // Verify that the parameters are actual properties/context defintions.
      if (!array_key_exists($key, $tool_definition['context_definitions'])) {
        return ExecutableResult::failure($this->t('Parameter %key is not a valid context definition for tool %tool_id.', [
          '%key' => $key,
          '%tool_id' => $tool_id,
        ]));
      }
    }
    $yaml[$tool_key] = [
      'tool' => $tool_id,
      'label' => $tool_name,
      'description' => $tool_description,
      'parameters' => $parameters,
    ];
    // Verify that the parameters are actual properties/context defintions.
    // Yaml encode it back.
    $tools_yaml = Yaml::dump($yaml, 2, 10);
    $agent->set('default_information_tools', $tools_yaml);

    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent after adding default information tool: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t('Default Information Tool with ID %tool_key has been added to agent %agent_id.', [
      '%tool_key' => $tool_key,
      '%agent_id' => $agent_id,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
