<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:add_property_definitions_to_tool',
  label: new TranslatableMarkup('Add Property Definitions To Tool'),
  description: new TranslatableMarkup('This can add one property definition to a specific tool on a specific agent by agent ID. This should only be used when needing to set specific hardcoded values on a property or overriding the property description.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to add a tool to."),
      required: TRUE,
    ),
    'tool_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool ID"),
      description: new TranslatableMarkup("The machine name of the tool you want to addd from the agent."),
      required: TRUE,
    ),
    'property_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Property ID"),
      description: new TranslatableMarkup("The machine name of the property definition you want to add to the tool."),
      required: TRUE,
    ),
    'restriction_type' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Restriction Type"),
      description: new TranslatableMarkup("The type of restriction to apply. Can be 'only_allow' or 'force_value'. If left empty all values are allowed."),
      required: FALSE,
    ),
    'restriction_values' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Restriction Values"),
      description: new TranslatableMarkup("A newline separated list of values to use for the restriction. If force_value is used, only one value should be provided."),
      required: FALSE,
    ),
    'property_description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Property Description"),
      description: new TranslatableMarkup("An optional override description for the property when added to the agent. If left empty, the default property description will be used. Defaults to empty."),
      required: FALSE,
    ),
  ],
)]
class AddPropertyDefinitionsToTool extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The tool plugin manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $toolManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->toolManager = $container->get('plugin.manager.ai.function_calls');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'tool_id' => $tool_id,
      'property_id' => $property_id,
      'restriction_type' => $restriction_type,
      'restriction_values' => $restriction_values,
      'property_description' => $property_description,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load the agent.
    /** @var \Drupal\ai_agents\AiAgentInterface $agent */
    $agent = $agent_storage->load($agent_id);
    if (!$agent) {
      return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
    }

    // Check if the tool exists.
    $tool_definition = $this->toolManager->getDefinition($tool_id);
    if (!$tool_definition) {
      return ExecutableResult::failure($this->t('Tool with ID %tool_id not found.', ['%tool_id' => $tool_id]));
    }

    // Check so the property definition exists on the tool.
    foreach ($tool_definition['context_definitions'] as $key => $context_definition) {
      if ($key === $property_id) {
        // Found the property definition.
        $property_definition = $context_definition;
        break;
      }
    }
    if (!isset($property_definition)) {
      return ExecutableResult::failure($this->t('Property definition with ID %property_id not found on tool %tool_id.', [
        '%property_id' => $property_id,
        '%tool_id' => $tool_id,
      ]));
    }

    // Check if the tool exists on the agent.
    $tools = $agent->get('tools');
    if (!isset($tools[$tool_id])) {
      return ExecutableResult::failure($this->t('Tool with ID %tool_id not found on agent %agent_id.', [
        '%tool_id' => $tool_id,
        '%agent_id' => $agent_id,
      ]));
    }

    // Load the tool usage limits.
    $usage_limits = $agent->get('tool_usage_limits');
    $usage_limits[$tool_id][$property_id] = [
      'action' => $restriction_type ?: '',
      'values' => $restriction_values ? explode("\n", $restriction_values) : [],
    ];
    $agent->set('tool_usage_limits', $usage_limits);

    // Load the tool settings.
    $tool_settings = $agent->get('tool_settings');
    $tool_settings[$tool_id]['property_description_override'][$property_id] = $property_description ?: '';
    $agent->set('tool_settings', $tool_settings);

    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent after adding property to tool: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t('Property definition with ID %property_id has been added to tool %tool_id on agent %agent_id.', [
      '%property_id' => $property_id,
      '%tool_id' => $tool_id,
      '%agent_id' => $agent_id,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
