<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation for the tool used to load agent information.
 */
#[Tool(
  id: 'ai_agent_agent:get_agent_info',
  label: new TranslatableMarkup('Get Agent Info'),
  description: new TranslatableMarkup('This will get the agent information about one or many agents.'),
  operation: ToolOperation::Read,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want more information about. If left empty, all agents will be returned."),
      required: FALSE,
    ),
    'verbose' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Verbose Output"),
      description: new TranslatableMarkup("By default only ID and name is given back. If verbose is true, all information about the agent will be returned."),
      required: FALSE,
    ),
  ],
)]
class GetAgentInfo extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'verbose' => $verbose,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    $agents = [];
    if ($agent_id) {
      $agent = $agent_storage->load($agent_id);
      if ($agent) {
        $agents[] = $agent;
      }
      // If the agent ID is provided but not found, still return success.
      else {
        return ExecutableResult::success($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
      }
    }
    else {
      $agents = $agent_storage->loadMultiple();
    }

    $result = [];
    foreach ($agents as $agent) {
      if ($verbose) {
        $agent_data = $agent->toArray();
        $result[] = $agent_data;
      }
      else {
        $result[] = [
          'id' => $agent->id(),
          'name' => $agent->label(),
        ];
      }
    }
    return ExecutableResult::success($this->t("The following Agents was found:\n\n @agents", [
      '@agents' => Yaml::dump($result, 4, 10),
    ]), ['result' => $result]);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
