<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation for the tool used to load tool information.
 */
#[Tool(
  id: 'ai_agent_agent:get_tool_info',
  label: new TranslatableMarkup('Get Tool Info'),
  description: new TranslatableMarkup('This will get the tool information about one or many tools.'),
  operation: ToolOperation::Read,
  input_definitions: [
    'tool_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool ID"),
      description: new TranslatableMarkup("The machine name of the tool you want more information about. If left empty, all tools will be returned."),
      required: FALSE,
    ),
    'verbose' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Verbose Output"),
      description: new TranslatableMarkup("By default only ID and name is given back. If verbose is true, all information about the tools will be returned."),
      required: FALSE,
    ),
  ],
  output_definitions: [
    'result' => new InputDefinition(
      data_type: 'list',
      label: new TranslatableMarkup("Tool Information"),
      description: new TranslatableMarkup("The information about the tool or tools requested."),
      required: TRUE,
    ),
  ],
)]
class GetToolInfo extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * The tool plugin manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $toolManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->toolManager = $container->get('plugin.manager.ai.function_calls');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'tool_id' => $tool_id,
      'verbose' => $verbose,
    ] = $values;

    $tools = [];
    if ($tool_id) {
      $tool = $this->toolManager->getDefinition($tool_id);
      if ($tool) {
        $tools[] = $tool;
      }
      // If the tool ID is provided but not found, still return success.
      else {
        return ExecutableResult::success($this->t('Tool with ID %id not found.', ['%id' => $tool_id]));
      }
    }
    else {
      $tools = $this->toolManager->getDefinitions();
    }

    $results = [];
    foreach ($tools as $tool) {
      // Skip all tools starting with action_plugin: as those are internal.
      if (str_starts_with($tool['id'], 'action_plugin:')) {
        continue;
      }
      $result = [
        'id' => $tool['id'],
        'name' => (string) $tool['name'],
        'description' => (string) $tool['description'],
      ];
      if ($verbose) {
        $result += $tool;
        foreach ($tool['context_definitions'] as $key => $context_definition) {
          $result['context_definitions'][$key] = [
            'label' => (string) $context_definition->getLabel(),
            'description' => (string) $context_definition->getDescription(),
            'data_type' => $context_definition->getDataType(),
            'required' => $context_definition->isRequired(),
            'default_value' => $context_definition->getDefaultValue(),
          ];
        }
      }
      $results[] = $result;
    }
    return ExecutableResult::success($this->t("The following tool was found:\n\n @tools", [
      '@tools' => Yaml::dump($results, 4, 10),
    ]), ['result' => $results]);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
