<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:modify_agent',
  label: new TranslatableMarkup('Modify Agent Tool'),
  description: new TranslatableMarkup('This can create or modify agents that already exists.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'operation' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Operation"),
      description: new TranslatableMarkup("The operation to perform: create or update."),
      required: TRUE,
    ),
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to create or update."),
      required: TRUE,
    ),
    'name' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent Name"),
      description: new TranslatableMarkup("The human readable name of the agent. Only required when creating a new agent. If left empty when updating an existing agent, the name will not be changed."),
      required: FALSE,
    ),
    'description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Description"),
      description: new TranslatableMarkup("The human readable description of the agent. Only required when creating a new agent. If left empty when updating an existing agent, the description will not be changed."),
      required: FALSE,
    ),
    'prompt' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent Prompt"),
      description: new TranslatableMarkup("The human readable instruction prompt of the agent. Only required when creating a new agent. If left empty when updating an existing agent, the prompt will not be changed. The instruction prompt is the prompt to be used in a ReACT agent."),
      required: FALSE,
    ),
    'max_loops' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Max Loops"),
      description: new TranslatableMarkup("The maximum number of loops the agent is allowed to perform. If left empty when updating an existing agent, the max loops will not be changed. The default is 20."),
      required: FALSE,
    ),
  ],
  output_definitions: [
    'result' => new InputDefinition(
      data_type: 'array',
      label: new TranslatableMarkup("Save Aget Info"),
      description: new TranslatableMarkup("The information about the tool or tools requested."),
      required: TRUE,
    ),
  ],
)]
class ModifyAgent extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'name' => $name,
      'description' => $description,
      'prompt' => $prompt,
      'operation' => $operation,
      'max_loops' => $max_loops,
    ] = $values;

    // If opertion is create, name, description and prompt are required.
    if ($operation == 'create' && (empty($name) || empty($description) || empty($prompt))) {
      return ExecutableResult::failure($this->t('When creating a new agent, name, description and prompt are required.'));
    }

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load existing agent or create a new one.
    if ($operation == 'update') {
      /** @var \Drupal\ai_agents\AiAgentInterface $agent */
      $agent = $agent_storage->load($agent_id);
      if (!$agent) {
        return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
      }
    }
    else {
      /** @var \Drupal\ai_agents\AiAgentInterface $agent */
      $agent = $agent_storage->create([
        'id' => $agent_id,
        'label' => $name,
        'tools' => [],
        'orchestration_agent' => FALSE,
        'triage_agent' => FALSE,
        'max_loops' => $max_loops ?? 20,
      ]);
    }

    // Set the fields if provided.
    if (!empty($name)) {
      $agent->set('label', $name);
    }
    if (!empty($description)) {
      $agent->set('description', $description);
    }
    if (!empty($prompt)) {
      $agent->set('system_prompt', $prompt);
    }
    if (!empty($max_loops)) {
      $agent->set('max_loops', $max_loops);
    }
    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t("The following agent was created:\n\n @agent", [
      '@agent' => Yaml::dump($agent->toArray(), 4, 10),
    ]), ['result' => $agent->toArray()]);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
