<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:remove_tool',
  label: new TranslatableMarkup('Remove Agent Tool'),
  description: new TranslatableMarkup('This can remove one agent tool from a specific agent by agent ID.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to remove a tool from."),
      required: TRUE,
    ),
    'tool_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool ID"),
      description: new TranslatableMarkup("The machine name of the tool you want to remove from the agent."),
      required: TRUE,
    ),
  ],
)]
class RemoveAgentTool extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'tool_id' => $tool_id,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load the agent.
    /** @var \Drupal\ai_agents\AiAgentInterface $agent */
    $agent = $agent_storage->load($agent_id);
    if (!$agent) {
      return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
    }

    // Check if the tool exists on the agent.
    $tools = $agent->get('tools');
    if (!isset($tools[$tool_id])) {
      return ExecutableResult::failure($this->t('Tool with ID %tool_id not found on agent %agent_id and can not be removed.', [
        '%tool_id' => $tool_id,
        '%agent_id' => $agent_id,
      ]));
    }

    // Remove the tool from the agent.
    unset($tools[$tool_id]);
    $agent->set('tools', $tools);

    // Remove tool settings if they exist.
    $tool_settings = $agent->get('tool_settings');
    if (isset($tool_settings[$tool_id])) {
      unset($tool_settings[$tool_id]);
      $agent->set('tool_settings', $tool_settings);
    }

    // Remove tool usage limits if they exist.
    $tool_usage_limits = $agent->get('tool_usage_limits');
    if (isset($tool_usage_limits[$tool_id])) {
      unset($tool_usage_limits[$tool_id]);
      $agent->set('tool_usage_limits', $tool_usage_limits);
    }
    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent after removing tool: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t('Tool with ID %tool_id has been removed from agent %agent_id.', [
      '%tool_id' => $tool_id,
      '%agent_id' => $agent_id,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
