<?php

declare(strict_types=1);

namespace Drupal\ai_agent_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation for the tool used to save an agent.
 */
#[Tool(
  id: 'ai_agent_agent:remove_default_information_tool',
  label: new TranslatableMarkup('Remove Default Information Tool'),
  description: new TranslatableMarkup('This can remove one default information tool from a specific agent by agent ID.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'agent_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent ID"),
      description: new TranslatableMarkup("The machine name of the agent you want to remove a tool from."),
      required: TRUE,
    ),
    'tool_key' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Tool Key"),
      description: new TranslatableMarkup("The key of the default information tool you want to remove from the agent. Not the tool id."),
      required: TRUE,
    ),
  ],
)]
class RemoveDefaultInformationTool extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'agent_id' => $agent_id,
      'tool_key' => $tool_key,
    ] = $values;

    // Get the agent storage.
    $agent_storage = $this->entityTypeManager->getStorage('ai_agent');

    // Load the agent.
    /** @var \Drupal\ai_agents\AiAgentInterface $agent */
    $agent = $agent_storage->load($agent_id);
    if (!$agent) {
      return ExecutableResult::failure($this->t('Agent with ID %id not found.', ['%id' => $agent_id]));
    }

    // Check if the tool exists on the agent.
    $tools = $agent->get('default_information_tools');
    // Yaml decode it.
    $yaml = Yaml::parse($tools);

    // If its not found, return failure.
    if (!isset($yaml[$tool_key])) {
      return ExecutableResult::failure($this->t('Tool with key %tool_key not found on agent %agent_id.', [
        '%tool_key' => $tool_key,
        '%agent_id' => $agent_id,
      ]));
    }

    // Remove the tool.
    unset($yaml[$tool_key]);
    // Yaml encode it back.
    $tools_yaml = Yaml::dump($yaml, 2, 10);
    $agent->set('default_information_tools', $tools_yaml);

    // Save the agent.
    try {
      $agent->save();
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Failed to save agent after removing tool: %message', ['%message' => $e->getMessage()]));
    }

    return ExecutableResult::success($this->t('Tool with ID %tool_key has been removed from agent %agent_id.', [
      '%tool_key' => $tool_key,
      '%agent_id' => $agent_id,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs administer ai_agent permission to use this tool.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer ai_agent',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
