<?php

namespace Drupal\alt_text_lab\Controller;

use Drupal\alt_text_lab\Service\AltTextLabApi;
use Drupal\alt_text_lab\Config\AltTextLabUrls;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Bulk generation dashboard page.
 */
class AltTextLabBulkController extends ControllerBase
{

  protected AltTextLabApi $api;
  protected EntityFieldManagerInterface $entityFieldManager;
  protected Connection $database;

  public static function create(ContainerInterface $container)
  {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->api = $container->get('alt_text_lab.api');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->database = $container->get('database');
    return $instance;
  }

  /**
   * Bulk generation dashboard.
   */
  public function page(): array
  {
    // Generic stats for media image fields with enabled alt attribute.
    [$total, $without_alt] = $this->calculateMediaImageStats();

    $build['#attached']['library'][] = 'alt_text_lab/bulk_page';

    $build['wrapper_open'] = [
      '#markup' => '<div class="alt-text-lab-bulk-wrapper">',
    ];

    $build['stats'] = $this->buildStatsCards($total, $without_alt);

    // Credits / account info block.
    $build['account'] = $this->buildAccountInfoBlock();

    // Info about target images.
    $build['info'] = [
      '#markup' => '<p class="alt-text-lab-bulk-info">' . $this->t('Alt text will be generated only for @count images without alt text.', ['@count' => $without_alt]) . '</p>',
    ];

    $has_targets = $total > 0;

    $build['form'] = $this->buildGenerationForm('alt_text_lab.bulk_run', 'alt_text_lab.bulk_run', $has_targets, $total, $without_alt);

    $build['wrapper_close'] = [
      '#markup' => '</div>',
    ];

    return $build;
  }

  /**
   * Handle bulk generation run: enqueue items and redirect to history.
   */
  public function run()
  {
    $request = \Drupal::request();
    $token = (string)$request->request->get('form_token');

    if (!\Drupal::service('csrf_token')->validate($token, 'alt_text_lab.bulk_run')) {
      throw $this->createAccessDeniedException();
    }

    $overwrite = (bool)$request->request->get('overwrite');
    $targets = $this->collectBulkTargets($overwrite);

    if (empty($targets)) {
      $this->messenger()->addStatus($this->t('No images found for bulk generation.'));
      return $this->redirect('alt_text_lab.bulk');
    }

    // Create bulk run record.
    $bulk_id = $this->database->insert('alt_text_lab_bulk_history')
      ->fields([
        'count_total' => count($targets),
        'created' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();

    // Enqueue all items.
    $queue = \Drupal::queue('alt_text_lab_bulk_queue');
    foreach ($targets as $target) {
      $target['bulk_id'] = (int)$bulk_id;
      $queue->createItem($target);
    }

    $this->messenger()->addStatus($this->t('Bulk generation has been started for @count images.', ['@count' => count($targets)]));

    return $this->redirect('alt_text_lab.history_bulk');
  }

  /**
   * Calculates statistics for media image fields with enabled alt attribute.
   */
  protected function calculateMediaImageStats(): array
  {
    $total = 0;
    $without_alt = 0;

    $media_bundles = $this->entityTypeManager()
      ->getStorage('media_type')
      ->loadMultiple();

    if (empty($media_bundles)) {
      return [$total, $without_alt];
    }

    $schema = $this->database->schema();

    foreach ($media_bundles as $bundle_id => $bundle) {
      $definitions = $this->entityFieldManager->getFieldDefinitions('media', $bundle_id);

      foreach ($definitions as $field_name => $definition) {
        if ($definition->getType() !== 'image') {
          continue;
        }

        $field_config_name = sprintf('field.field.media.%s.%s', $bundle_id, $field_name);
        $field_config = $this->config($field_config_name);
        if (!$field_config || !$field_config->get('settings.alt_field')) {
          continue;
        }

        $table = 'media__' . $field_name;
        $alt_column = $field_name . '_alt';

        if (!$schema->tableExists($table)) {
          continue;
        }

        $total_query = $this->database->select($table, 't');
        $total_query->addExpression('COUNT(*)', 'cnt');
        $table_total = (int)$total_query->execute()->fetchField();

        $no_alt_query = $this->database->select($table, 't');
        $or = $no_alt_query->orConditionGroup()
          ->isNull($alt_column)
          ->condition($alt_column, '', '=');
        $no_alt_query->condition($or);
        $no_alt_query->addExpression('COUNT(*)', 'cnt');
        $table_without_alt = (int)$no_alt_query->execute()->fetchField();

        $total += $table_total;
        $without_alt += $table_without_alt;
      }
    }

    return [$total, $without_alt];
  }

  /**
   * Collect all media image items that need generation.
   */
  protected function collectBulkTargets(bool $overwrite = FALSE): array
  {
    $targets = [];

    $media_bundles = $this->entityTypeManager()
      ->getStorage('media_type')
      ->loadMultiple();

    if (empty($media_bundles)) {
      return $targets;
    }

    foreach ($media_bundles as $bundle_id => $bundle) {
      $definitions = $this->entityFieldManager->getFieldDefinitions('media', $bundle_id);

      $image_fields = [];
      foreach ($definitions as $field_name => $definition) {
        if ($definition->getType() !== 'image') {
          continue;
        }
        $field_config_name = sprintf('field.field.media.%s.%s', $bundle_id, $field_name);
        $field_config = $this->config($field_config_name);
        if (!$field_config || !$field_config->get('settings.alt_field')) {
          continue;
        }
        $image_fields[] = $field_name;
      }

      if (empty($image_fields)) {
        continue;
      }

      $media_storage = $this->entityTypeManager()->getStorage('media');
      $media_list = $media_storage->loadByProperties(['bundle' => $bundle_id]);

      foreach ($media_list as $media) {
        foreach ($image_fields as $field_name) {
          if (!$media->hasField($field_name) || $media->get($field_name)->isEmpty()) {
            continue;
          }

          foreach ($media->get($field_name) as $delta => $item) {
            $fid = (int)($item->target_id ?? 0);
            if (!$fid) {
              continue;
            }

            $alt_current = isset($item->alt) ? (string)$item->alt : '';
            if (!$overwrite && $alt_current !== '') {
              continue;
            }

            $targets[] = [
              'fid' => $fid,
              'media_id' => (int)$media->id(),
              'field_name' => $field_name,
              'delta' => (int)$delta,
            ];
          }
        }
      }
    }

    return $targets;
  }

  /**
   * Display statistics and generation form for selected media items.
   */
  public function selection(): array
  {
    $tempstore = \Drupal::service('tempstore.private')->get('alt_text_lab_bulk_selection');
    $media_ids = $tempstore->get('media_ids');

    if (empty($media_ids) || !is_array($media_ids)) {
      $this->messenger()->addWarning($this->t('No media items selected. Please select items from the media library first.'));
      return [
        '#markup' => '<p>' . $this->t('No items selected.') . '</p>',
      ];
    }

    // Calculate statistics for selected media.
    [$total_images, $without_alt] = $this->calculateSelectionStats($media_ids);

    $has_targets = $total_images > 0;

    $build['#attached']['library'][] = 'alt_text_lab/bulk_page';

    $build['wrapper_open'] = [
      '#markup' => '<div class="alt-text-lab-bulk-wrapper">',
    ];

    $build['info'] = [
      '#markup' => '<p>' . $this->t('You have selected @count media items.', ['@count' => count($media_ids)]) . '</p>',
    ];

    $build['stats'] = $this->buildStatsCards($total_images, $without_alt);

    // Credits / account info block.
    $build['account'] = $this->buildAccountInfoBlock();

    // Info about target images.
    $build['info2'] = [
      '#markup' => '<p class="alt-text-lab-bulk-info">' . $this->t('Alt text will be generated only for @count images without alt text.', ['@count' => $without_alt]) . '</p>',
    ];

    // Simple form that triggers bulk run.
    $build['form'] = $this->buildGenerationForm('alt_text_lab.bulk_selection_run', 'alt_text_lab.bulk_selection_run', $has_targets, $total_images, $without_alt, TRUE);

    $build['wrapper_close'] = [
      '#markup' => '</div>',
    ];

    return $build;
  }

  /**
   * Run bulk generation for selected media items.
   */
  public function runSelection()
  {
    $request = \Drupal::request();
    $token = (string)$request->request->get('form_token');

    if (!\Drupal::service('csrf_token')->validate($token, 'alt_text_lab.bulk_selection_run')) {
      throw $this->createAccessDeniedException();
    }

    $tempstore = \Drupal::service('tempstore.private')->get('alt_text_lab_bulk_selection');
    $media_ids = $tempstore->get('media_ids');

    if (empty($media_ids) || !is_array($media_ids)) {
      $this->messenger()->addWarning($this->t('No media items selected.'));
      return $this->redirect('alt_text_lab.bulk');
    }

    $overwrite = (bool)$request->request->get('overwrite');

    $targets = $this->collectSelectionTargets($media_ids, $overwrite);

    if (empty($targets)) {
      $this->messenger()->addStatus($this->t('No images found for bulk generation.'));
      $tempstore->delete('media_ids');
      return $this->redirect('alt_text_lab.bulk');
    }

    $bulk_id = $this->database->insert('alt_text_lab_bulk_history')
      ->fields([
        'count_total' => count($targets),
        'created' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();

    // Enqueue all items.
    $queue = \Drupal::queue('alt_text_lab_bulk_queue');
    foreach ($targets as $target) {
      $target['bulk_id'] = (int)$bulk_id;
      $queue->createItem($target);
    }

    $this->messenger()->addStatus($this->t('Bulk generation has been started for @count images.', ['@count' => count($targets)]));
    $tempstore->delete('media_ids');

    return $this->redirect('alt_text_lab.history_bulk');
  }

  /**
   * Calculate statistics for selected media items.
   */
  protected function calculateSelectionStats(array $media_ids): array
  {
    $total = 0;
    $without_alt = 0;

    if (empty($media_ids)) {
      return [$total, $without_alt];
    }

    $media_storage = $this->entityTypeManager()->getStorage('media');
    $media_list = $media_storage->loadMultiple($media_ids);

    $schema = $this->database->schema();

    foreach ($media_list as $media) {
      $bundle_id = $media->bundle();
      $definitions = $this->entityFieldManager->getFieldDefinitions('media', $bundle_id);

      foreach ($definitions as $field_name => $definition) {
        if ($definition->getType() !== 'image') {
          continue;
        }

        $field_config_name = sprintf('field.field.media.%s.%s', $bundle_id, $field_name);
        $field_config = $this->config($field_config_name);
        if (!$field_config || !$field_config->get('settings.alt_field')) {
          continue;
        }

        $table = 'media__' . $field_name;
        $alt_column = $field_name . '_alt';

        if (!$schema->tableExists($table)) {
          continue;
        }

        $total_query = $this->database->select($table, 't');
        $total_query->condition('entity_id', $media->id(), '=');
        $total_query->addExpression('COUNT(*)', 'cnt');
        $table_total = (int)$total_query->execute()->fetchField();

        $no_alt_query = $this->database->select($table, 't');
        $no_alt_query->condition('entity_id', $media->id(), '=');
        $or = $no_alt_query->orConditionGroup()
          ->isNull($alt_column)
          ->condition($alt_column, '', '=');
        $no_alt_query->condition($or);
        $no_alt_query->addExpression('COUNT(*)', 'cnt');
        $table_without_alt = (int)$no_alt_query->execute()->fetchField();

        $total += $table_total;
        $without_alt += $table_without_alt;
      }
    }

    return [$total, $without_alt];
  }

  /**
   * Collect targets from selected media items.
   */
  protected function collectSelectionTargets(array $media_ids, bool $overwrite = FALSE): array
  {
    $targets = [];

    if (empty($media_ids)) {
      return $targets;
    }

    $media_storage = $this->entityTypeManager()->getStorage('media');
    $media_list = $media_storage->loadMultiple($media_ids);

    $schema = $this->database->schema();

    foreach ($media_list as $media) {
      $bundle_id = $media->bundle();
      $definitions = $this->entityFieldManager->getFieldDefinitions('media', $bundle_id);

      foreach ($definitions as $field_name => $definition) {
        if ($definition->getType() !== 'image') {
          continue;
        }

        $field_config_name = sprintf('field.field.media.%s.%s', $bundle_id, $field_name);
        $field_config = $this->config($field_config_name);
        if (!$field_config || !$field_config->get('settings.alt_field')) {
          continue;
        }

        $table = 'media__' . $field_name;
        $target_id_column = $field_name . '_target_id';
        $alt_column = $field_name . '_alt';

        if (!$schema->tableExists($table)) {
          continue;
        }

        // Query database directly for selected media items
        $query = $this->database->select($table, 't');
        $query->condition('entity_id', $media->id(), '=');
        $query->fields('t', ['delta', $target_id_column, $alt_column]);
        $results = $query->execute()->fetchAll();

        foreach ($results as $row) {
          $fid = (int)($row->{$target_id_column} ?? 0);
          if (!$fid) {
            continue;
          }

          $alt_current = isset($row->{$alt_column}) ? (string)$row->{$alt_column} : '';
          // Check if alt is empty (NULL or empty string)
          if (empty($alt_current)) {
            $alt_current = '';
          }

          if (!$overwrite && $alt_current !== '') {
            continue;
          }

          $targets[] = [
            'fid' => $fid,
            'media_id' => (int)$media->id(),
            'field_name' => $field_name,
            'delta' => (int)$row->delta,
          ];
        }
      }
    }

    return $targets;
  }

  /**
   * Build statistics cards (total images / without alt).
   */
  protected function buildStatsCards(int $total, int $without_alt): array
  {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['alt-text-lab-bulk-cards']],
      'total' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
        'label' => ['#markup' => '<strong>' . $this->t('Count of images') . '</strong>'],
        'value' => ['#markup' => '<div>' . $total . '</div>'],
      ],
      'without_alt' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
        'label' => ['#markup' => '<strong>' . $this->t('Images without alt text') . '</strong>'],
        'value' => ['#markup' => '<div>' . $without_alt . '</div>'],
      ],
    ];
  }

  /**
   * Build generation form with CSRF token.
   */
  protected function buildGenerationForm(string $token_key, string $route_name, bool $has_targets, int $total, int $without_alt, bool $show_cancel = FALSE): array
  {
    $token = \Drupal::service('csrf_token')->get($token_key);
    $action = Url::fromRoute($route_name)->toString();

    $config = $this->config('alt_text_lab.settings');
    $api_key = (string)($config->get('api_key') ?? '');
    $has_valid_api_key = FALSE;

    if ($api_key !== '') {
      [$account, $error] = $this->api->getAccountInfo($api_key);

      if (is_array($account) && !empty($account['isActive'])) {
        $has_valid_api_key = TRUE;
      }
    }

    $form_enabled = $has_targets && $has_valid_api_key;

    $cancel_button = $show_cancel ? '<a href="/admin/content/media" class="button">{{ cancel_label }}</a>' : '';

    return [
      '#type' => 'inline_template',
      '#template' => '<form method="post" action="{{ action }}" class="alt-text-lab-bulk-form" data-total="{{ total }}" data-without-alt="{{ without_alt }}">
              <input type="hidden" name="form_token" value="{{ token }}" />
              <div class="alt-text-lab-bulk-form__controls">
                <label>
                  <input type="checkbox" name="overwrite" value="1" {% if not form_enabled %}disabled="disabled"{% endif %} />
                  {{ overwrite_label }}
                </label>
              </div>
              {% if form_enabled %}
                <button type="submit" class="button button--primary">{{ button_label }}</button>
              {% else %}
                <button type="button" class="button" disabled="disabled">{{ disabled_label }}</button>
              {% endif %}
              ' . $cancel_button . '
            </form>',
      '#context' => [
        'action' => $action,
        'token' => $token,
        'button_label' => $this->t('Run generation'),
        'disabled_label' => $this->t('Run generation'),
        'overwrite_label' => $this->t('Overwrite existing alt text'),
        'cancel_label' => $this->t('Cancel'),
        'has_targets' => $has_targets,
        'form_enabled' => $form_enabled,
        'total' => $total,
        'without_alt' => $without_alt,
      ],
    ];
  }

  /**
   * Build account info / credits block.
   */
  protected function buildAccountInfoBlock(): array
  {
    $config = $this->config('alt_text_lab.settings');
    $api_key = (string)($config->get('api_key') ?? '');

    $account = NULL;
    $is_valid_account = FALSE;
    $credits = 0;
    $api_error = NULL;

    if ($api_key !== '') {
      [$account, $api_error] = $this->api->getAccountInfo($api_key);
      if (is_array($account)) {
        $is_valid_account = !empty($account['isActive']);
        $credits = (int)($account['credits'] ?? 0);
      }
    }

    // No API key set.
    if ($api_key === '') {
      $settings_url = Url::fromRoute('alt_text_lab.settings')->toString();
      return [
        '#markup' => '<div class="messages messages--warning alt-text-lab-welcome">'
          . '<p><strong>' . $this->t('Welcome to Alt Text Lab!') . '</strong></p>'
          . '<p>' . $this->t('To start generating alt text, please set your API key in the module settings.') . '</p>'
          . '<p><a href="' . $settings_url . '">' . $this->t('Open module settings →') . '</a></p>'
          . '</div>',
      ];
    }

    // Invalid API key – non-2xx from API.
    if ($account === FALSE) {
      return [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['messages', 'messages--error'],
        ],
        'content' => [
          '#markup' => $this->t('Your API key is invalid. Please check your API key or <a href=":url" target="_blank">create a new API key</a>.', [
            ':url' => AltTextLabUrls::apiKeysInvalid(),
          ]),
        ],
      ];
    }

    // Could not load account info (other error).
    if ($api_error) {
      $settings_url = Url::fromRoute('alt_text_lab.settings')->toString();
      return [
        '#markup' => '<div class="messages messages--warning alt-text-lab-welcome">'
          . '<p>' . $this->t('Could not load account information. Please verify your API key.') . '</p>'
          . '<p><a href="' . $settings_url . '">' . $this->t('Open module settings →') . '</a></p>'
          . '</div>',
      ];
    }

    // No credits left.
    if (!$is_valid_account || $credits <= 0) {
      return [
        '#markup' => '<div class="messages messages--warning"><p>'
          . $this->t('You have no more credits left. To bulk update your library, please check your subscription on AltTextLab.')
          . '</p></div>',
      ];
    }

    // Show available credits.
    return [
      '#markup' => '<div class="messages messages--status"><p>'
        . $this->t('Available credits: @credits', ['@credits' => $credits])
        . '</p></div>',
    ];
  }

}
