<?php

namespace Drupal\alt_text_lab\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Drupal\alt_text_lab\Config\AltTextLabUrls;
use Drupal\alt_text_lab\Service\AltTextLabApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Dashboard page for AltTextLab.
 */
class AltTextLabDashboardController extends ControllerBase {

  protected AltTextLabApi $api;
  protected DateFormatterInterface $dateFormatter;

  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->api = $container->get('alt_text_lab.api');
    $instance->dateFormatter = $container->get('date.formatter');
    return $instance;
  }

  /**
   * Page callback.
   */
  public function content(): array {
    $build = [];

    $build['#attached']['library'][] = 'alt_text_lab/dashboard_page';
    $build['#attached']['library'][] = 'alt_text_lab/bulk_page';

    $config = $this->config('alt_text_lab.settings');
    $api_key = (string) $config->get('api_key');

    $account = NULL;
    $api_error = NULL;

    if ($api_key !== '') {
      [$account, $api_error] = $this->api->getAccountInfo($api_key);
    }

    // Alerts & account info.
    $alerts = [];

    if ($api_key === '') {
      // No API key configured.
      $alerts[] = [
        'type' => 'warning',
        'extra_class' => 'alt-text-lab-welcome',
        'markup' => $this->t('<p class="alt-font-medium">@welcome</p><p>@hint</p><a href=":url">@link_label</a>', [
          '@welcome' => $this->t('Welcome to Alt Text Lab!'),
          '@hint' => $this->t('To start generating alt text, please set your API key in the module settings.'),
          ':url' => Url::fromRoute('alt_text_lab.settings')->toString(),
          '@link_label' => $this->t('Open module settings →'),
        ]),
      ];
    }
    elseif ($api_error) {
      // Transport / decoding error when talking to API.
      $alerts[] = [
        'type' => 'error',
        'markup' => $this->t('Failed to fetch account information: @message', ['@message' => $api_error]),
      ];
    }
    elseif ($account === FALSE) {
      // Invalid API key – non-2xx from API.
      $alerts[] = [
        'type' => 'error',
        'markup' => $this->t('Your API key is invalid. Please check your API key or <a href=":url" target="_blank">create a new API key</a>.', [
          ':url' => AltTextLabUrls::apiKeysInvalid(),
        ]),
      ];
    }
    elseif (is_array($account)) {
      // Subscription status.
      if (!empty($account['isActive'])) {
        $plan = $account['plan'] ?? NULL;
        $alerts[] = [
          'type' => 'success',
          'markup' => $this->t('You’re on the <strong>@plan</strong> plan.', ['@plan' => $plan ?: $this->t('unknown')]),
        ];
      }
      else {
        $alerts[] = [
          'type' => 'warning',
          'markup' => $this->t('<strong>You don’t have an active subscription.</strong>'),
        ];
      }

      // Credits / disabled reasons.
      if (!empty($account['isActive']) && (int) ($account['credits'] ?? 0) === 0) {
        $alerts[] = [
          'type' => 'warning',
          'markup' => $this->t('<strong>Alt text generation is currently disabled due to zero credits on your account.</strong>'),
        ];
      }
      elseif (empty($account['isActive'])) {
        $alerts[] = [
          'type' => 'warning',
          'markup' => $this->t('Alt text generation is currently <strong>disabled</strong>.'),
        ];
      }
    }

    if (!empty($alerts)) {
      $build['alerts'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['alt-text-lab-dashboard-messages']],
        '#weight' => 1,
      ];

      foreach ($alerts as $delta => $alert) {
        $type = $alert['type'];
        if ($type === 'success') {
          $message_type = 'status';
        }
        elseif ($type === 'error') {
          $message_type = 'error';
        }
        else {
          $message_type = 'warning';
        }

        $classes = ['messages', 'messages--' . $message_type];
        if (!empty($alert['extra_class'])) {
          $classes[] = $alert['extra_class'];
        }

        $build['alerts']['message_' . $delta] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => $classes,
          ],
          'content' => [
            '#markup' => $alert['markup'],
          ],
        ];
      }
    }

    // Simple statistics based on module tables.
    try {
      $connection = \Drupal::database();

      $query = $connection->select('alt_text_lab_generation_history', 'h');
      $query->addExpression('COUNT(*)', 'cnt');
      $total_generated = (int) $query->execute()->fetchField();

      $query = $connection->select('alt_text_lab_log', 'l');
      $query->addExpression('COUNT(*)', 'cnt');
      $total_failed = (int) $query->execute()->fetchField();

      $query = $connection->select('alt_text_lab_generation_history', 'h');
      $query->addExpression('COUNT(DISTINCT media_id)', 'cnt');
      $query->condition('media_id', 0, '>');
      $media_with_alt = (int) $query->execute()->fetchField();

      $query = $connection->select('alt_text_lab_bulk_history', 'b');
      $query->addExpression('COUNT(*)', 'cnt');
      $bulk_runs = (int) $query->execute()->fetchField();

      $build['stats'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['alt-text-lab-dashboard-stats', 'alt-text-lab-bulk-cards']],
        '#weight' => 2,
        'generated' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
          'label' => ['#markup' => '<strong>' . $this->t('Alt text generations') . '</strong>'],
          'value' => ['#markup' => '<div>' . $total_generated . '</div>'],
        ],
        'failed' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
          'label' => ['#markup' => '<strong>' . $this->t('Failed generations') . '</strong>'],
          'value' => ['#markup' => '<div>' . $total_failed . '</div>'],
        ],
        'media' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
          'label' => ['#markup' => '<strong>' . $this->t('Media items with generated alt text') . '</strong>'],
          'value' => ['#markup' => '<div>' . $media_with_alt . '</div>'],
        ],
        'bulk_runs' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['alt-text-lab-bulk-card']],
          'label' => ['#markup' => '<strong>' . $this->t('Bulk runs') . '</strong>'],
          'value' => ['#markup' => '<div>' . $bulk_runs . '</div>'],
        ],
      ];
    }
    catch (\Exception $e) {
      \Drupal::logger('alt_text_lab')->warning('Failed to load dashboard statistics: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    $build['intro'] = [
      '#markup' => '<p>' . $this->t('Overview of your Alt Text Lab subscription and module status.') . '</p>',
      '#weight' => 0,
    ];

    if (is_array($account)) {
      $credits = (int) ($account['credits'] ?? 0);
      $next_receiving = $account['nextReceiving'] ?? NULL;
      $next_receiving_at = $account['nextReceivingAt'] ?? NULL;

      $details_markup = '<div class="alt-text-lab-dashboard__summary">';
      $details_markup .= '<p>' . $this->t('Credits on your account: <strong>@credits</strong>', ['@credits' => $credits]) . '</p>';

      if (!empty($account['isActive']) && $next_receiving && $next_receiving_at) {
        $timestamp = strtotime($next_receiving_at);
        $formatted_date = $this->dateFormatter->format($timestamp, 'custom', 'm.d.Y');

        $details_markup .= '<p>' . $this->t('<strong>Next credits refill:</strong> @amount credits on <strong>@date</strong>', [
          '@amount' => $next_receiving,
          '@date' => $formatted_date,
        ]) . '</p>';
      }

      $details_markup .= '</div>';

      $build['account_details'] = [
        '#markup' => $details_markup,
        '#weight' => 3,
      ];

      // Upsell / subscription hints.
      if (!empty($account['isActive'])) {
        $build['plan_link'] = [
          '#markup' => '<p>' . $this->t('You can <a href=":url" target="_blank">upgrade your plan</a> to get more credits.', [
            ':url' => AltTextLabUrls::subscriptionDashboard(),
          ]) . '</p>',
          '#weight' => 4,
        ];
      }
      else {
        $build['plan_link'] = [
          '#markup' => '<p>' . $this->t('To unlock alt text generation, please <a href=":url" target="_blank">activate a subscription plan</a>.', [
            ':url' => AltTextLabUrls::subscriptionDashboard(),
          ]) . '</p>',
          '#weight' => 4,
        ];
      }
    }

    $build['settings_link'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['alt-text-lab-settings-actions']],
      '#weight' => 5,
      'button' => [
        '#type' => 'link',
        '#title' => $this->t('Open module settings'),
        '#url' => Url::fromRoute('alt_text_lab.settings'),
        '#attributes' => [
          'class' => ['button', 'alt-text-lab-settings-button'],
        ],
      ],
    ];

    return $build;
  }

}
