<?php

namespace Drupal\alt_text_lab\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Shows a simple table of successful alt text generations.
 */
class AltTextLabHistoryController extends ControllerBase {
  protected const ITEMS_PER_PAGE = 10;

  protected DateFormatterInterface $dateFormatter;
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->dateFormatter = $container->get('date.formatter');
    $instance->fileUrlGenerator = $container->get('file_url_generator');
    return $instance;
  }

  public function content(): array {
    $build['#attached']['library'][] = 'alt_text_lab/bulk_page';

    $header = [
      'preview' => $this->t('Image'),
      'current_alt' => $this->t('Current alt text'),
      'generated_alt' => $this->t('Generated alt text'),
      'created' => $this->t('Generated at'),
      'actions' => $this->t('Actions'),
    ];

    $limit = self::ITEMS_PER_PAGE;

    $request = \Drupal::request();
    $bulk_id = (int) $request->query->get('bulk_id');

    // Count total for pager.
    $count_query = \Drupal::database()->select('alt_text_lab_generation_history', 'h');
    $count_query->addExpression('COUNT(*)');
    if ($bulk_id > 0) {
      $count_query->condition('bulk_id', $bulk_id);
    }
    $total = (int) $count_query->execute()->fetchField();

    // Initialize pager.
    $pager = \Drupal::service('pager.manager')->createPager($total, $limit);
    $current_page = $pager->getCurrentPage();
    $offset = $current_page * $limit;

    $query = \Drupal::database()->select('alt_text_lab_generation_history', 'h')
      ->fields('h', ['id', 'fid', 'media_id', 'alt_text', 'created', 'field_name', 'delta'])
      ->orderBy('id', 'DESC')
      ->range($offset, $limit);
    if ($bulk_id > 0) {
      $query->condition('bulk_id', $bulk_id);
    }

    $result = $query->execute()->fetchAllAssoc('id');

    $rows = [];
    if ($result) {
      $fids = array_column($result, 'fid');
      $files = $this->entityTypeManager()->getStorage('file')->loadMultiple($fids);
      $media_ids = array_column($result, 'media_id');

      // Reset cache to get fresh data.
      $this->entityTypeManager()->getStorage('media')->resetCache($media_ids);
      $media_entities = $this->entityTypeManager()->getStorage('media')->loadMultiple($media_ids);

      foreach ($result as $record) {
        // Default cell content when file is missing.
        $preview_cell = [
          'data' => $this->t('Not available'),
        ];
        $has_media_link = FALSE;

        if (isset($files[$record->fid])) {
          $file = $files[$record->fid];
          $uri = $file->getFileUri();
          $image_url = $this->fileUrlGenerator->generateAbsoluteString($uri);

          $media_url = NULL;
          if (!empty($record->media_id) && isset($media_entities[$record->media_id])) {
            $media_url = Url::fromRoute('entity.media.edit_form', ['media' => $record->media_id])->toString();
          }

          $preview_cell = [
            'data' => [
              '#type' => 'inline_template',
              '#template' => '{% if edit_url %}<a href="{{ edit_url }}"><img src="{{ src }}" alt="" style="max-width:75px; max-height:75px;" /></a>{% else %}<img src="{{ src }}" alt="" style="max-width:75px; max-height:75px;" />{% endif %}',
              '#context' => [
                'edit_url' => $media_url,
                'src' => $image_url,
              ],
            ],
          ];

          if (!empty($media_url)) {
            $has_media_link = TRUE;
          }
        }

        // Current alt text - get directly from DB if we have field_name.
        $current_alt = '';
        if (!empty($record->media_id) && !empty($record->field_name)) {
          // Load media to get bundle for table name.
          $media = $media_entities[$record->media_id] ?? NULL;
          if ($media) {
            $bundle = $media->bundle();
            $field_name = $record->field_name;
            $delta = (int) ($record->delta ?? 0);
            $table_name = 'media__' . $field_name;

            // Query directly from field data table.
            try {
              $alt = \Drupal::database()->select($table_name, 'f')
                ->fields('f', [$field_name . '_alt'])
                ->condition('entity_id', $record->media_id)
                ->condition('delta', $delta)
                ->execute()
                ->fetchField();

              if ($alt !== FALSE) {
                $current_alt = (string) $alt;
              }
            }
            catch (\Exception $e) {
              \Drupal::logger('alt_text_lab')->debug('Could not read current alt text from field table @table: @message', [
                '@table' => $table_name,
                '@message' => $e->getMessage(),
              ]);
            }
          }
        }

        $current_cell = $current_alt;

        $action_cell = [
          'data' => $this->t('No media relation available'),
        ];

        if (!empty($record->media_id) && !empty($record->field_name) && $has_media_link) {
          $url_options = [];
          if (!empty($record->bulk_id)) {
            $url_options['query']['bulk_id'] = $record->bulk_id;
          }

          $action_cell['data'] = Link::fromTextAndUrl(
            $this->t('Use this alt text'),
            Url::fromRoute('alt_text_lab.history_use_alt', ['id' => $record->id], $url_options)
          )->toRenderable();
        }

        if (!$has_media_link || empty($record->media_id) || empty($record->field_name)) {
          $current_cell = [
            'data' => [
              '#type' => 'html_tag',
              '#tag' => 'span',
              '#value' => $this->t('No media relation'),
              '#attributes' => ['class' => ['alt-text-lab-warning']],
            ],
          ];

          $action_cell = [
            'data' => [
              '#type' => 'html_tag',
              '#tag' => 'span',
              '#value' => $this->t('Unavailable'),
              '#attributes' => ['class' => ['alt-text-lab-unavailable']],
            ],
          ];
        }

        $rows[] = [
          'preview' => $preview_cell,
          'current_alt' => $current_cell,
          'generated_alt' => $record->alt_text,
          'created' => $this->dateFormatter->format((int) $record->created, 'short'),
          'actions' => $action_cell,
        ];
      }
    }

    if (empty($rows)) {
      $build['empty'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('No alt text has been generated yet.') . '</div>',
      ];
      return $build;
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No items found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Failed generations tab.
   */
  public function failed(): array {
    $header = [
      'id' => [
        'data' => $this->t('ID'),
        'class' => ['text-center'],
      ],
      'preview' => $this->t('Image'),
      'message' => $this->t('Message'),
      'created' => [
        'data' => $this->t('Logged at'),
        'class' => ['text-center'],
      ],
    ];

    $limit = self::ITEMS_PER_PAGE;

    $request = \Drupal::request();
    $bulk_id = (int) $request->query->get('bulk_id');

    // Count total for pager.
    $count_query = \Drupal::database()->select('alt_text_lab_log', 'l');
    $count_query->addExpression('COUNT(*)');
    if ($bulk_id > 0) {
      $count_query->condition('bulk_id', $bulk_id);
    }
    $total = (int) $count_query->execute()->fetchField();

    // Initialize pager.
    $pager = \Drupal::service('pager.manager')->createPager($total, $limit);
    $current_page = $pager->getCurrentPage();
    $offset = $current_page * $limit;

    $query = \Drupal::database()->select('alt_text_lab_log', 'l')
      ->fields('l', ['id', 'fid', 'media_id', 'field_name', 'delta', 'message', 'created'])
      ->orderBy('id', 'DESC')
      ->range($offset, $limit);
    if ($bulk_id > 0) {
      $query->condition('bulk_id', $bulk_id);
    }

    $result = $query->execute()->fetchAllAssoc('id');

    $rows = [];
    if ($result) {
      $fids = array_column($result, 'fid');
      $files = $this->entityTypeManager()->getStorage('file')->loadMultiple($fids);
      $media_ids = array_column($result, 'media_id');
      $this->entityTypeManager()->getStorage('media')->resetCache($media_ids);
      $media_entities = $this->entityTypeManager()->getStorage('media')->loadMultiple($media_ids);

      foreach ($result as $record) {
        // Default cell content when file is missing.
        $preview_cell = [
          'data' => $this->t('Not available'),
        ];

        if (isset($files[$record->fid])) {
          $file = $files[$record->fid];
          $uri = $file->getFileUri();
          $image_url = $this->fileUrlGenerator->generateAbsoluteString($uri);

          $media_url = NULL;
          if (!empty($record->media_id) && isset($media_entities[$record->media_id])) {
            $media_url = Url::fromRoute('entity.media.edit_form', ['media' => $record->media_id])->toString();
          }

          $preview_cell = [
            'data' => [
              '#type' => 'inline_template',
              '#template' => '{% if edit_url %}<a href="{{ edit_url }}"><img src="{{ src }}" alt="" style="max-width:75px; max-height:75px;" /></a>{% else %}<img src="{{ src }}" alt="" style="max-width:75px; max-height:75px;" /></a>{% endif %}',
              '#context' => [
                'edit_url' => $media_url,
                'src' => $image_url,
              ],
            ],
          ];
        }

        $rows[] = [
          'id' => [
            'data' => $record->id,
            'class' => ['text-center'],
          ],
          'preview' => $preview_cell,
          'message' => [
            'data' => $record->message,
          ],
          'created' => [
            'data' => $this->dateFormatter->format((int) $record->created, 'short'),
            'class' => ['text-center'],
          ],
        ];
      }
    }

    if (empty($rows)) {
      $build['empty'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('No failed generations recorded yet.') . '</div>',
      ];
      return $build;
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No items found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Bulk generations history tab.
   */
  public function bulk(): array {
    $build['#attached']['library'][] = 'alt_text_lab/bulk_history';
    $build['#attached']['library'][] = 'alt_text_lab/bulk_page';

    $header = [
      'id' => $this->t('ID'),
      'total' => $this->t('Images to generate'),
      'completed' => $this->t('Completed'),
      'progress' => $this->t('Progress'),
      'success' => $this->t('Successfully generated'),
      'failed' => $this->t('Failed to generate'),
      'created' => $this->t('Generation date'),
    ];

    $limit = self::ITEMS_PER_PAGE;
    $connection = \Drupal::database();

    // Count total for pager.
    $count_query = $connection->select('alt_text_lab_bulk_history', 'b');
    $count_query->addExpression('COUNT(*)');
    $total = (int) $count_query->execute()->fetchField();

    // Initialize pager.
    $pager = \Drupal::service('pager.manager')->createPager($total, $limit);
    $current_page = $pager->getCurrentPage();
    $offset = $current_page * $limit;

    $query = $connection->select('alt_text_lab_bulk_history', 'b')
      ->fields('b', ['id', 'count_total', 'created'])
      ->orderBy('id', 'DESC')
      ->range($offset, $limit);

    $runs = $query->execute()->fetchAllAssoc('id');

    if (empty($runs)) {
      $build['empty'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('No bulk generations have been started yet.') . '</div>',
      ];
      return $build;
    }

    $build['notice'] = [
      '#markup' => '<div class="messages messages--status alt-text-lab-bulk-notice">' . $this->t('During active bulk generations, data on this page is automatically updated every @seconds seconds to reflect the current status.', ['@seconds' => 5]) . '</div>',
      '#weight' => -10,
    ];

    $bulk_ids = array_keys($runs);

    // Successful counts per bulk run.
    $success_query = $connection->select('alt_text_lab_generation_history', 'h')
      ->fields('h', ['bulk_id']);
    $success_query->addExpression('COUNT(*)', 'count');
    $success_query->condition('bulk_id', $bulk_ids, 'IN');
    $success_query->groupBy('bulk_id');
    $success_result = $success_query->execute()->fetchAllKeyed();

    // Failed counts per bulk run.
    $failed_query = $connection->select('alt_text_lab_log', 'l')
      ->fields('l', ['bulk_id']);
    $failed_query->addExpression('COUNT(*)', 'count');
    $failed_query->condition('bulk_id', $bulk_ids, 'IN');
    $failed_query->groupBy('bulk_id');
    $failed_result = $failed_query->execute()->fetchAllKeyed();

    // Build table rows with proper row attributes.
    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => [],
      '#empty' => $this->t('No items found.'),
    ];

    foreach ($runs as $run) {
      $success = (int) ($success_result[$run->id] ?? 0);
      $failed = (int) ($failed_result[$run->id] ?? 0);
      $completed = $success + $failed;
      $total = (int) $run->count_total;

      $percent = $total > 0 ? (int) round($completed / $total * 100) : 100;

      $success_cell = ['data' => $success];
      if ($success > 0) {
        $success_cell['data'] = Link::fromTextAndUrl((string) $success, Url::fromRoute('alt_text_lab.history', [], [
          'query' => ['bulk_id' => $run->id],
        ]))->toRenderable();
      }

      $failed_cell = ['data' => $failed];
      if ($failed > 0) {
        $failed_cell['data'] = Link::fromTextAndUrl((string) $failed, Url::fromRoute('alt_text_lab.history_failed', [], [
          'query' => ['bulk_id' => $run->id],
        ]))->toRenderable();
      }

      $build['table']['#rows'][] = [
        'data' => [
          ['data' => $run->id],
          ['data' => $total],
          ['data' => $this->t('@done/@total (@percent%)', [
            '@done' => $completed,
            '@total' => $total,
            '@percent' => $percent,
          ])],
          [
            'data' => [
              '#type' => 'inline_template',
              '#template' => '
                <div class="alt-text-lab-progress-cell">
                  <progress value="{{ completed }}" max="{{ total }}" class="alt-text-lab-progress"></progress>
                  <div class="alt-text-lab-progress-label">{{ percent }}%</div>
                </div>',
              '#context' => [
                'completed' => $completed,
                'total' => $total ?: 1,
                'percent' => $percent,
              ],
            ],
          ],
          $success_cell,
          $failed_cell,
          ['data' => $this->dateFormatter->format((int) $run->created, 'short')],
        ],
        'class' => ['alt-text-lab-bulk-row'],
        'data-total' => $total,
        'data-completed' => $completed,
        'data-bulk-id' => $run->id,
      ];
    }

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * AJAX endpoint for bulk runs status polling.
   *
   * Returns counts of successful and failed generations for given bulk run IDs.
   */
  public function bulkStatus(): JsonResponse {
    $request = \Drupal::request();
    $payload = json_decode($request->getContent() ?? '', TRUE);
    $ids = [];

    if (is_array($payload) && !empty($payload['ids']) && is_array($payload['ids'])) {
      foreach ($payload['ids'] as $id) {
        $id = (int) $id;
        if ($id > 0) {
          $ids[] = $id;
        }
      }
    }

    if (empty($ids)) {
      return new JsonResponse([
        'success' => TRUE,
        'data' => [],
      ]);
    }

    $connection = \Drupal::database();

    // Load bulk runs to get total counts.
    $runs_query = $connection->select('alt_text_lab_bulk_history', 'b')
      ->fields('b', ['id', 'count_total'])
      ->condition('id', $ids, 'IN');
    $runs = $runs_query->execute()->fetchAllAssoc('id');

    if (empty($runs)) {
      return new JsonResponse([
        'success' => TRUE,
        'data' => [],
      ]);
    }

    // Successful counts per bulk run.
    $success_query = $connection->select('alt_text_lab_generation_history', 'h')
      ->fields('h', ['bulk_id']);
    $success_query->addExpression('COUNT(*)', 'count');
    $success_query->condition('bulk_id', array_keys($runs), 'IN');
    $success_query->groupBy('bulk_id');
    $success_result = $success_query->execute()->fetchAllKeyed();

    // Failed counts per bulk run.
    $failed_query = $connection->select('alt_text_lab_log', 'l')
      ->fields('l', ['bulk_id']);
    $failed_query->addExpression('COUNT(*)', 'count');
    $failed_query->condition('bulk_id', array_keys($runs), 'IN');
    $failed_query->groupBy('bulk_id');
    $failed_result = $failed_query->execute()->fetchAllKeyed();

    $data = [];
    foreach ($runs as $run) {
      $id = (int) $run->id;
      $success = (int) ($success_result[$id] ?? 0);
      $failed = (int) ($failed_result[$id] ?? 0);

      $data[] = [
        'id' => $id,
        'successfulCount' => $success,
        'failedCount' => $failed,
        'count_total' => (int) $run->count_total,
      ];
    }

    return new JsonResponse([
      'success' => TRUE,
      'data' => $data,
    ]);
  }

  /**
   * Applies generated alt text to the corresponding media entity.
   */
  public function useAlt(int $id) {
    $connection = \Drupal::database();
    $record = $connection->select('alt_text_lab_generation_history', 'h')
      ->fields('h', ['id', 'media_id', 'alt_text', 'field_name', 'delta', 'bulk_id'])
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    $query = [];
    if ($record && !empty($record->bulk_id)) {
      $query['bulk_id'] = (int) $record->bulk_id;
    }

    if (!$record) {
      $this->messenger()->addError($this->t('History record not found.'));
      return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
    }

    if (empty($record->media_id) || empty($record->field_name)) {
      $this->messenger()->addError($this->t('Cannot apply alt text: media reference is missing.'));
      return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
    }

    $media = $this->entityTypeManager()->getStorage('media')->load($record->media_id);
    if (!$media) {
      $this->messenger()->addError($this->t('Media entity not found.'));
      return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
    }

    if (!$media->hasField($record->field_name)) {
      $this->messenger()->addError($this->t('Media field @field was not found on this entity.', ['@field' => $record->field_name]));
      return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
    }

    $delta = (int) $record->delta;
    $items = $media->get($record->field_name);
    if (!isset($items[$delta])) {
      $this->messenger()->addError($this->t('Image item with delta @delta was not found.', ['@delta' => $delta]));
      return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
    }

    $items[$delta]->set('alt', $record->alt_text);

    try {
      $media->save();
      $this->messenger()->addStatus($this->t('Generated alt text has been applied to the media item.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to apply alt text: @message', ['@message' => $e->getMessage()]));
    }

    return $this->redirect('alt_text_lab.history', [], ['query' => $query]);
  }

}


