<?php

namespace Drupal\alt_text_lab\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\file\Entity\File;
use Drupal\alt_text_lab\Service\AltTextGenerator;
use Drupal\alt_text_lab\Service\AltTextLabLogger;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for generating alt text via AJAX.
 */
class GenerateAltTextController extends ControllerBase {

  protected AltTextGenerator $generator;
  protected AltTextLabLogger $logger;

  public static function create(ContainerInterface $container) {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->generator = $container->get('alt_text_lab.generator');
    $instance->logger = $container->get('alt_text_lab.logger');
    return $instance;
  }

  /**
   * Generate alt text for a file.
   */
  public function generate($file_id) {
    // Load the file entity.
    $file = File::load($file_id);
    if (!$file) {
      $this->logger->logFailure((int) $file_id, 'File not found.');
      return new JsonResponse([
        'error' => $this->t('File not found.'),
      ], 404);
    }

    // Check access.
    if (!$file->access('view')) {
      $this->logger->logFailure((int) $file_id, 'Access denied.');
      return new JsonResponse([
        'error' => $this->t('Access denied.'),
      ], 403);
    }

    // Use the generator service to generate alt text.
    $result = $this->generator->generate($file);

    if ($result['error']) {
      $this->logger->logFailure((int) $file_id, $result['error']);

      // Map error codes to HTTP status codes.
      $status_code = 500;
      if ($result['error_code'] === 'API_KEY_IS_INVALID') {
        $status_code = 401;
      }
      elseif ($result['error_code'] === 'NOT_ENOUGH_FUNDS') {
        $status_code = 403;
      }
      elseif (strpos($result['error'], 'not an image') !== FALSE || strpos($result['error'], 'validation') !== FALSE) {
        $status_code = 400;
      }

      return new JsonResponse([
        'error' => $this->t($result['error']),
      ], $status_code);
    }

    $alt_text = $result['alt_text'];

    try {
      $history_id = \Drupal::database()->insert('alt_text_lab_generation_history')
        ->fields([
          'fid' => (int) $file_id,
          'media_id' => 0,
          'field_name' => NULL,
          'delta' => 0,
          'alt_text' => $alt_text,
          'created' => \Drupal::time()->getRequestTime(),
          'bulk_id' => 0,
        ])
        ->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('alt_text_lab')->warning('Failed to log generation history for file @fid: @message', [
        '@fid' => $file_id,
        '@message' => $e->getMessage(),
      ]);
      $history_id = 0;
    }

    return new JsonResponse([
      'alt_text' => $alt_text,
      'history_id' => $history_id,
    ]);
  }

}


