<?php

namespace Drupal\alt_text_lab\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\alt_text_lab\Config\AltTextLabLanguages;
use Drupal\alt_text_lab\Config\AltTextLabModels;
use Drupal\alt_text_lab\Config\AltTextLabUrls;
use Drupal\alt_text_lab\Service\AltTextLabApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the AltTextLab settings form.
 */
class AltTextLabSettingsForm extends ConfigFormBase {

  protected AltTextLabApi $api;

  public static function create(ContainerInterface $container) {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->api = $container->get('alt_text_lab.api');
    return $instance;
  }

  public function getFormId(): string {
    return 'alt_text_lab_settings_form';
  }

  protected function getEditableConfigNames(): array {
    return ['alt_text_lab.settings'];
  }

  /**
   * Returns available AI models.
   */
  protected function getModels(): array {
    $models = AltTextLabModels::all();
    $translated = [];
    foreach ($models as $key => $label) {
      $translated[$key] = $this->t($label);
    }
    return $translated;
  }

  /**
   * Returns supported languages.
   */
  protected function getLanguages(): array {
    $languages = AltTextLabLanguages::all();
    $translated = [];
    foreach ($languages as $code => $label) {
      $translated[$code] = $this->t($label);
    }
    return $translated;
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('alt_text_lab.settings');
    $api_key = (string) ($config->get('api_key') ?? '');

    $description = '';

    $form['#attached']['library'][] = 'alt_text_lab/settings_page';

    if ($api_key === '') {
      $description .= ' ' . $this->t('Get your API key at <a href=":url" target="_blank" rel="noopener noreferrer">AltTextLab.com</a> &gt; API Keys &gt; Create API key.', [
        ':url' => AltTextLabUrls::apiKeysHint(),
      ]);
    } else {
      [$account, $error] = $this->api->getAccountInfo($api_key);

      if ($error) {
        $description .= ' ' . $this->t('We could not verify your API key right now: @message', ['@message' => $error]);
      } elseif ($account === FALSE) {
        $description .= ' ' . $this->t('<span class="alt-text-lab-api-status alt-text-lab-api-status--invalid">Your API key looks <strong>invalid</strong>. Please check your API key or <a href=":url" target="_blank" rel="noopener noreferrer">create a new API key</a>.</span>', [
          ':url' => AltTextLabUrls::apiKeysInvalid(),
        ]);
      } elseif (is_array($account)) {
        $credits = (int) ($account['credits'] ?? 0);
        $description .= ' ' . $this->t('<span class="alt-text-lab-api-status alt-text-lab-api-status--valid">Your API key is <strong>valid</strong>. Credits on your account: <strong>@credits</strong>.</span>', [
          '@credits' => $credits,
        ]);
      }
    }

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AltTextLab API key'),
      '#default_value' => $api_key,
      '#description' => $description,
      '#required' => FALSE,
    ];

    $form['on_upload_generate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-generate alt text for new media items'),
      '#description' => $this->t('Automatically generate alt text for new uploaded images.'),
      '#default_value' => $config->get('on_upload_generate') ?? TRUE,
    ];

    $form['model_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Model type'),
      '#options' => $this->getModels(),
      '#default_value' => $config->get('model_name') ?? NULL,
    ];

    $form['lang'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $this->getLanguages(),
      '#default_value' => $config->get('lang') ?? 'en',
    ];

    $form['is_public'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('This site is reachable over the public internet'),
      '#description' => $this->t('Allow AltTextLab to access your images via their URLs for faster and more efficient processing. If your site is private or hosted on localhost, you can disable this option and upload images directly instead — but keeping it enabled is recommended for best performance.'),
      '#default_value' => $config->get('is_public') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $values = $form_state->getValues();

    $this->configFactory->getEditable('alt_text_lab.settings')
      ->set('api_key', $values['api_key'] ?? '')
      ->set('on_upload_generate', (bool) ($values['on_upload_generate'] ?? FALSE))
      ->set('model_name', $values['model_name'] ?? NULL)
      ->set('lang', $values['lang'] ?? 'en')
      ->set('is_public', (bool) ($values['is_public'] ?? FALSE))
      ->save();
  }

}




